/* Copyright (c) Mark J. Kilgard, 1997. 
 * modified by Olivier.Arsac 1998 */

#include "U.h"
#include "GL/glu.h"
#include "texfont.h"


/* byte swap a 32-bit value */
#define SWAPL(x, n) { \
                 n = ((char *) (x))[0];\
                 ((char *) (x))[0] = ((char *) (x))[3];\
                 ((char *) (x))[3] = n;\
                 n = ((char *) (x))[1];\
                 ((char *) (x))[1] = ((char *) (x))[2];\
                 ((char *) (x))[2] = n; }

/* byte swap a short */
#define SWAPS(x, n) { \
                 n = ((char *) (x))[0];\
                 ((char *) (x))[0] = ((char *) (x))[1];\
                 ((char *) (x))[1] = n; }



/* this part is automaticaly updated, do NOT edit below */
/**************************************************************** Prototypes */

/* Don't change it without checking <linux/types.h>. */
/* Some synonyms used historically in the kernel and elsewhere */
/************************************************************ End Prototypes */
/* end of automaticaly updated part */
static TexGlyphVertexInfo *getTCVI(TexFont * txf, int c);

/************************************************************ End Prototypes */
/* end of automaticaly updated part */



Module TexGlyphVertexInfo *
getTCVI(TexFont * txf, int c)
{
  TexGlyphVertexInfo *tgvi;

  /* Automatically substitute uppercase letters with lowercase if not
     uppercase available (and vice versa). */
  if ((c >= txf->min_glyph) && (c < txf->min_glyph + txf->range)) {
    tgvi = txf->lut[c - txf->min_glyph];
    if (tgvi) {
      return tgvi;
    }
    if (islower(c)) {
      c = toupper(c);
      if ((c >= txf->min_glyph) && (c < txf->min_glyph + txf->range)) {
	return txf->lut[c - txf->min_glyph];
      }
    }
    if (isupper(c)) {
      c = tolower(c);
      if ((c >= txf->min_glyph) && (c < txf->min_glyph + txf->range)) {
	return txf->lut[c - txf->min_glyph];
      }
    }
  }
  /* no glyph available return first one (space) */
  return txf->lut[0];
}

Module char *lastError;

World char *
txfErrorString(void)
{
  return lastError;
}

World TexFont *
txfLoadFont(char *filename)
{
  TexFont *txf;
  FILE *file;
  GLfloat w, h, xstep, ystep;
  char fileid[4], tmp;
  unsigned char *texbitmap;
  int min_glyph, max_glyph;
  int endianness, swap, format, stride, width, height;
  int i, j;
  unsigned long got;

  txf = NULL;
  file = fopen(filename, "rb");
  if (file == NULL) {
    lastError = "file open failed.";
    goto error;
  }
  txf = (TexFont *) malloc(sizeof(TexFont));
  if (txf == NULL) {
    lastError = "out of memory.";
    goto error;
  }
  /* For easy cleanup in error case. */
  txf->tgi = NULL;
  txf->tgvi = NULL;
  txf->lut = NULL;
  txf->teximage = NULL;

  got = fread(fileid, 1, 4, file);
  if (got != 4 || strncmp(fileid, "\377txf", 4)) {
    lastError = "not a texture font file.";
    goto error;
  }
  /*CONSTANTCONDITION */
  UAssert(sizeof(int) == 4);	/* Ensure external file format size. */
  got = fread(&endianness, sizeof(int), 1, file);
  if (got == 1 && endianness == 0x12345678) {
    swap = 0;
  }
  else if (got == 1 && endianness == 0x78563412) {
    swap = 1;
  }
  else {
    lastError = "not a texture font file.";
    goto error;
  }
#define EXPECT(n) if (got != n) { lastError = "premature end of file."; goto error; }
  got = fread(&format, sizeof(int), 1, file);
  EXPECT(1);
  got = fread(&txf->tex_width, sizeof(int), 1, file);
  EXPECT(1);
  got = fread(&txf->tex_height, sizeof(int), 1, file);
  EXPECT(1);
  got = fread(&txf->max_ascent, sizeof(int), 1, file);
  EXPECT(1);
  got = fread(&txf->max_descent, sizeof(int), 1, file);
  EXPECT(1);
  got = fread(&txf->num_glyphs, sizeof(int), 1, file);
  EXPECT(1);

  if (swap) {
    SWAPL(&format, tmp);
    SWAPL(&txf->tex_width, tmp);
    SWAPL(&txf->tex_height, tmp);
    SWAPL(&txf->max_ascent, tmp);
    SWAPL(&txf->max_descent, tmp);
    SWAPL(&txf->num_glyphs, tmp);
  }
  txf->tgi = (TexGlyphInfo *) malloc(txf->num_glyphs * sizeof(TexGlyphInfo));
  if (txf->tgi == NULL) {
    lastError = "out of memory.";
    goto error;
  }
  /*CONSTANTCONDITION */
  UAssert(sizeof(TexGlyphInfo) == 12);	/* Ensure external file format size. */
  got = fread(txf->tgi, sizeof(TexGlyphInfo), txf->num_glyphs, file);
  EXPECT(txf->num_glyphs);

  if (swap) {
    for (i = 0; i < txf->num_glyphs; i++) {
      SWAPS(&txf->tgi[i].c, tmp);
      SWAPS(&txf->tgi[i].x, tmp);
      SWAPS(&txf->tgi[i].y, tmp);
    }
  }
  txf->tgvi = (TexGlyphVertexInfo *)
    malloc(txf->num_glyphs * sizeof(TexGlyphVertexInfo));
  if (txf->tgvi == NULL) {
    lastError = "out of memory.";
    goto error;
  }
  w = txf->tex_width;
  h = txf->tex_height;
  xstep = 0.5 / w;
  ystep = 0.5 / h;
  for (i = 0; i < txf->num_glyphs; i++) {
    TexGlyphInfo *tgi;

    tgi = &txf->tgi[i];
    txf->tgvi[i].t0[0] = tgi->x / w + xstep;
    txf->tgvi[i].t0[1] = tgi->y / h + ystep;
    txf->tgvi[i].v0[0] = tgi->xoffset;
    txf->tgvi[i].v0[1] = tgi->yoffset;
    txf->tgvi[i].t1[0] = (tgi->x + tgi->width) / w + xstep;
    txf->tgvi[i].t1[1] = tgi->y / h + ystep;
    txf->tgvi[i].v1[0] = tgi->xoffset + tgi->width;
    txf->tgvi[i].v1[1] = tgi->yoffset;
    txf->tgvi[i].t2[0] = (tgi->x + tgi->width) / w + xstep;
    txf->tgvi[i].t2[1] = (tgi->y + tgi->height) / h + ystep;
    txf->tgvi[i].v2[0] = tgi->xoffset + tgi->width;
    txf->tgvi[i].v2[1] = tgi->yoffset + tgi->height;
    txf->tgvi[i].t3[0] = tgi->x / w + xstep;
    txf->tgvi[i].t3[1] = (tgi->y + tgi->height) / h + ystep;
    txf->tgvi[i].v3[0] = tgi->xoffset;
    txf->tgvi[i].v3[1] = tgi->yoffset + tgi->height;
    txf->tgvi[i].advance = tgi->advance;
  }

  min_glyph = txf->tgi[0].c;
  max_glyph = txf->tgi[0].c;
  for (i = 1; i < txf->num_glyphs; i++) {
    if (txf->tgi[i].c < min_glyph) {
      min_glyph = txf->tgi[i].c;
    }
    if (txf->tgi[i].c > max_glyph) {
      max_glyph = txf->tgi[i].c;
    }
  }
  txf->min_glyph = min_glyph;
  txf->range = max_glyph - min_glyph + 1;

  txf->lut = (TexGlyphVertexInfo **)
    calloc(txf->range, sizeof(TexGlyphVertexInfo *));
  if (txf->lut == NULL) {
    lastError = "out of memory.";
    goto error;
  }
  for (i = 0; i < txf->num_glyphs; i++) {
    txf->lut[txf->tgi[i].c - txf->min_glyph] = &txf->tgvi[i];
  }

  switch (format) {
  case TXF_FORMAT_BYTE:{
      unsigned char *orig;

      orig = (unsigned char *) malloc(txf->tex_width * txf->tex_height);
      if (orig == NULL) {
	lastError = "out of memory.";
	goto error;
      }
      got = fread(orig, 1, txf->tex_width * txf->tex_height, file);
      EXPECT(txf->tex_width * txf->tex_height);
      txf->teximage = (unsigned char *)
	malloc(2 * txf->tex_width * txf->tex_height);
      if (txf->teximage == NULL) {
	lastError = "out of memory.";
	goto error;
      }
      for (i = 0; i < txf->tex_width * txf->tex_height; i++) {
	txf->teximage[i * 2] = orig[i];
	txf->teximage[i * 2 + 1] = orig[i];
      }
      free(orig);
      break;
    }
  case TXF_FORMAT_BITMAP:
    width = txf->tex_width;
    height = txf->tex_height;
    stride = (width + 7) >> 3;
    texbitmap = (unsigned char *) malloc(stride * height);
    if (texbitmap == NULL) {
      lastError = "out of memory.";
      goto error;
    }
    got = fread(texbitmap, 1, stride * height, file);
    EXPECT(stride * height);

    txf->teximage = (unsigned char *) calloc(width * height * 2, 1);
    if (txf->teximage == NULL) {
      lastError = "out of memory.";
      goto error;
    }
    for (i = 0; i < height; i++) {
      for (j = 0; j < width; j++) {
	if (texbitmap[i * stride + (j >> 3)] & (1 << (j & 7))) {
	  txf->teximage[(i * width + j) * 2] = 255;
	  txf->teximage[(i * width + j) * 2 + 1] = 255;
	}
      }
    }
    free(texbitmap);
    break;
  }
  txfEstablishTexture(txf);
  fclose(file);
  return txf;

error:

  if (txf) {
    if (txf->tgi)
      free(txf->tgi);
    if (txf->tgvi)
      free(txf->tgvi);
    if (txf->lut)
      free(txf->lut);
    if (txf->teximage)
      free(txf->teximage);
    free(txf);
  }
  if (file)
    fclose(file);
  return NULL;
}

World GLuint
txfEstablishTexture(TexFont * txf)
{
  glGenTextures(1, &txf->texobj);
  glBindTexture(GL_TEXTURE_2D, txf->texobj);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  gluBuild2DMipmaps(GL_TEXTURE_2D, GL_LUMINANCE_ALPHA,
		    txf->tex_width, txf->tex_height,
		    GL_LUMINANCE_ALPHA, GL_UNSIGNED_BYTE, txf->teximage);

  return txf->texobj;
}

World void
txfBindFontTexture(TexFont * txf)
{
  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, txf->texobj);
}

World void
txfUnloadFont(TexFont * txf)
{
  if (txf->teximage) {
    free(txf->teximage);
  }
  free(txf->tgi);
  free(txf->tgvi);
  free(txf->lut);
  free(txf);
}

World void
txfGetStringMetrics(TexFont * txf, char *string, int len, int *width, int *max_ascent, int *max_descent)
{
  TexGlyphVertexInfo *tgvi;
  int w, i;

  w = 0;
  for (i = 0; i < len; i++) {
    if (string[i] == 27) {
      switch (string[i + 1]) {
      case 'M':
	i += 4;
	break;
      case 'T':
	i += 7;
	break;
      case 'L':
	i += 7;
	break;
      case 'F':
	i += 13;
	break;
      }
    }
    else {
      tgvi = getTCVI(txf, string[i]);
      w += tgvi->advance;
    }
  }
  *width = w;
  *max_ascent = txf->max_ascent;
  *max_descent = txf->max_descent;
}

World Float32
txfRenderGlyph(TexFont * txf, Int32 c, Float32 angle)
{
  TexGlyphVertexInfo *tgvi;

  tgvi = getTCVI(txf, c);
  if (angle != 0) {
    Float32 cx, cy;
    glPushMatrix();
    cx = (tgvi->v3[0] - tgvi->v1[0]) / 2.;
    cy = (tgvi->v1[1] - tgvi->v3[1] ) / 2.;
        glTranslatef(-cx, -cy, 0);
    glRotatef(angle, 0, 0, 1);
       glTranslatef(cx, cy, 0);
  }
  glBegin(GL_QUADS);
  glTexCoord2fv(tgvi->t0);
  glVertex2sv(tgvi->v0);
  glTexCoord2fv(tgvi->t1);
  glVertex2sv(tgvi->v1);
  glTexCoord2fv(tgvi->t2);
  glVertex2sv(tgvi->v2);
  glTexCoord2fv(tgvi->t3);
  glVertex2sv(tgvi->v3);
  glEnd();
  if (angle != 0)
    glPopMatrix();
  glTranslatef(tgvi->advance, 0.0, 0.0);
  return tgvi->advance;
}

World Float32
txfRenderGlyphSmooth(TexFont * txf, int c, UByte * col1, UByte * col2)
{
  TexGlyphVertexInfo *tgvi;
  Float32 c1[] =
  {.2, .2, 1, 1};
  Float32 c2[] =
  {1, .2, .2, 1};

  tgvi = getTCVI(txf, c);
  glBegin(GL_QUADS);
  glColor4fv(c1);
  glTexCoord2fv(tgvi->t0);
  glVertex2sv(tgvi->v0);
  glTexCoord2fv(tgvi->t1);
  glVertex2sv(tgvi->v1);

  glColor4fv(c2);
  glTexCoord2fv(tgvi->t2);
  glVertex2sv(tgvi->v2);
  glTexCoord2fv(tgvi->t3);
  glVertex2sv(tgvi->v3);
  glEnd();

  glTranslatef(tgvi->advance, 0.0, 0.0);
  return tgvi->advance;
}

World Float32
txfRenderString(TexFont * txf, char *string)
{
  Float32 dx = 0;

  while (*string)
    dx += txfRenderGlyph(txf, *(string++), 0);
  return dx;
}

enum {
  MONO, TOP_BOTTOM, LEFT_RIGHT, FOUR
};

World void
txfRenderFancyString(TexFont * txf, char *string, int len)
{
  TexGlyphVertexInfo *tgvi;
  GLubyte c[4][3];
  int mode = MONO;
  int i;

  for (i = 0; i < len; i++) {
    if (string[i] == 27) {
      switch (string[i + 1]) {
      case 'M':
	mode = MONO;
	glColor3ubv((GLubyte *) & string[i + 2]);
	i += 4;
	break;
      case 'T':
	mode = TOP_BOTTOM;
	memcpy(c, &string[i + 2], 6);
	i += 7;
	break;
      case 'L':
	mode = LEFT_RIGHT;
	memcpy(c, &string[i + 2], 6);
	i += 7;
	break;
      case 'F':
	mode = FOUR;
	memcpy(c, &string[i + 2], 12);
	i += 13;
	break;
      }
    }
    else {
      switch (mode) {
      case MONO:
	txfRenderGlyph(txf, string[i], 0);
	break;
      case TOP_BOTTOM:
	tgvi = getTCVI(txf, string[i]);
	glBegin(GL_QUADS);
	glColor3ubv(c[0]);
	glTexCoord2fv(tgvi->t0);
	glVertex2sv(tgvi->v0);
	glTexCoord2fv(tgvi->t1);
	glVertex2sv(tgvi->v1);
	glColor3ubv(c[1]);
	glTexCoord2fv(tgvi->t2);
	glVertex2sv(tgvi->v2);
	glTexCoord2fv(tgvi->t3);
	glVertex2sv(tgvi->v3);
	glEnd();
	glTranslatef(tgvi->advance, 0.0, 0.0);
	break;
      case LEFT_RIGHT:
	tgvi = getTCVI(txf, string[i]);
	glBegin(GL_QUADS);
	glColor3ubv(c[0]);
	glTexCoord2fv(tgvi->t0);
	glVertex2sv(tgvi->v0);
	glColor3ubv(c[1]);
	glTexCoord2fv(tgvi->t1);
	glVertex2sv(tgvi->v1);
	glColor3ubv(c[1]);
	glTexCoord2fv(tgvi->t2);
	glVertex2sv(tgvi->v2);
	glColor3ubv(c[0]);
	glTexCoord2fv(tgvi->t3);
	glVertex2sv(tgvi->v3);
	glEnd();
	glTranslatef(tgvi->advance, 0.0, 0.0);
	break;
      case FOUR:
	tgvi = getTCVI(txf, string[i]);
	glBegin(GL_QUADS);
	glColor3ubv(c[0]);
	glTexCoord2fv(tgvi->t0);
	glVertex2sv(tgvi->v0);
	glColor3ubv(c[1]);
	glTexCoord2fv(tgvi->t1);
	glVertex2sv(tgvi->v1);
	glColor3ubv(c[2]);
	glTexCoord2fv(tgvi->t2);
	glVertex2sv(tgvi->v2);
	glColor3ubv(c[3]);
	glTexCoord2fv(tgvi->t3);
	glVertex2sv(tgvi->v3);
	glEnd();
	glTranslatef(tgvi->advance, 0.0, 0.0);
	break;
      }
    }
  }
}

World int
txfInFont(TexFont * txf, int c)
{
  /* NOTE: No uppercase/lowercase substituion. */
  if ((c >= txf->min_glyph) && (c < txf->min_glyph + txf->range)) {
    if (txf->lut[c - txf->min_glyph]) {
      return 1;
    }
  }
  return 0;
}
