/*
 * This program is under the GNU GPL.
 * Use at your own risk.
 *
 * This small OpenGL demo code plots parametric surfaces with some fancy 
 * features. It is not intended to be used as a real ploter but rather 
 * as a small demo of what OpenGL can do. It has been designed to run 
 * with fast OpenGL engines (a PII450/voodoo2 gives me a 78 fps in plain 
 * surface mode and a 12.5 fps in the last mode (fog, 4 mirror images,
 * projection on walls, grids, clip plane, banner, textures, ...)
 *
 */

/*
 * FEATURES:
 *
 *  - multiple mirror images (with progressive fade)
 *  - visible transparent clip plane for surfaces
 *  - visible transparent normals
 *  - visible shining lights (true spherical non projective)
 *  - progressive lens flares (fading when hidden by surface)
 *  - correct surface occlusion detection for lights (even in mirrorss)
 *  - realistic wall lighting
 *  - surface projection on walls
 *  - stupid scrolling banner (wall wrapping)
 *  - textures on surface (generated or pictures)
 *  - texture fonts (sucked from X server)
 *  - antialiased adaptative grids
 *  - fog for depth cuing
 *  - animated surface, observer, lamp, clip plane...
 *  - animation speeds are frame rate independant
 *  - predefined display modes
 *    (from my old Apple][+ greenish point plot to
 *    mirror/fog/texture... voodoo2)
 *  - predefined parametric surfaces (z=f(x,y) and (x,y,z)=f(u,v))
 *  - user input for parameters controling surfaces
 *  - unified PNG I/O for images/textures.
 *  - stats (triangle/second and frame/second)
 *  - comprehensive help screen ;)
 *
 *
 * written by Olivier ARSAC
 *            Olivier.Arsac@inria.fr
 *            http://www.inria.fr/safir/oarsac/
 */

/* TODO:

 *  - add st sound support for real men demos ;)
 *  - cast shadows from surface (on itself and walls)
 *  - environment/chrome mapping
 *  - apply aplha (trimed) textures on surface to cut holes
 *  - do antialiasing and/or focal blur when no animation 
 *    in progress. (alas accumulation buffer is not usable on v2)
 *  - fade/blast from one surface to another
 *  - virtual ball (ouch need a deep rework)
 *  - find a workaround for texture evaluators...  
 *    (glide2/mesa3 turns to soft engine with that enabled)
 *    alas needed to draw fair Z isolines because z color
 *    interpolation is loosy :(
 *  - use remaining walls to draw (useful?) things
 *  - add C evaluator for direct user input of functions
 *  - display values of user parameters 
 *    (and surface equations?)
 *  - add some kind of numerical ticks?
 *  - profile for speed optimisation (vertex arrays?)
 *  - find more stupid ideas to add in TODO
 *  - stop adding two stupid ideas to this list each time
 *    I manage to remove one ;)
 */

/*
 * BUGS (known ;) :
 *
 *  - overlap of texture fonts lead to flickers due to
 *    Zbuffer artifacts. use a  glDepthFunc(GL_ALWAYS)
 *    and clip-plane in walls to solve that.
 *    (glColorMask(tttf) is not usable on mesa3.0/glide2/v2)
 *  - light mirror zbuffer troubles when Z(light)~=0
 *    need to clip light quad :( 
 *    (fixed that by keeping light from crossing ground)
 *  - normal for some surfaces are inverted 
 *    (not easy to fix on klein bottle ;)
 *  - too slow o:)
 *  - make a cleaner distrib
 *  - add more comments
 */


#include <GL/glut.h>
#ifdef XMESA
#include <GL/xmesa.h>
#endif

#include "U.h"
#include "png.h"
#include "pngio.h"
#include "texfont.h"
#include "dwchrono.h"


/* 
 * Physical parameters and various const
 */

#define ObsDefFov            90
#define ObsRhoSpeedInc       (1./10)
#define ObsThetaSpeedInc     (Pi/30)
#define ObsPhiSpeedInc       (Pi/40)

#define LightNb              1
#define LightThetaSpeedInc   (Pi/20)
#define LightPhiSpeedInc     (Pi/20)
#define LightFadeSpeedInc    1.5
#define LampRadius           .05
#define LampResolution       10
#define LampDistance         .8

#define PlaneRhoSpeedInc     0.15
#define PlaneThetaSpeedInc   (Pi/30.)

#define NormEpsilon          0.0001

#define WallResolution       12
#define ClipPlaneResolution  6

#define MirrorTranspInc      0.50
#define ParameterSpeedInc    0.1

#define FrameStats           20
#define FrameRateCount       5	// fps averaged on last 5 frames

#define SurfaceDefaultResolution 30
#define CR1                  SurfaceDefaultResolution
#define SurfaceResMult       1.2
#define SurfaceSpeedInc      0.1


/* sorry, you have to be sure you don't use more */
#define MaxSurfaceNb         32
#define MaxParameterNb       16
#define MaxFontNb            8
#define MaxFlareNb           16
#define MaxShineNb           16

/* lens flare textures */
#define TexFlareNb           6
#define TexShineNb           10

#define	TexImageDim          256

#define HelpMargin           0.05

#define FontHelpFile         "textures/fonts/default.txf"
#define FontBannerFile       "textures/fonts/curlfont.txf"
#define BannerScale          .05	/* 20 chars per wall */
#define BannerSpeed          90	/* nb font points scroll/s */

/*
 * Types and Structures
 */

typedef enum MaterialId {
  MatDefault = 0, MatSurfaceFront, MatSurfaceBack, MatLight, MatPerfectMirror, MatMirror, MatWall, MatLine, MatSurfaceProj, MatGold, MatClipPlane, MatBanner, MatLast
} MaterialId;

typedef enum WallId {
  WallGround = 0, WallLeft, WallRight, WallTop, WallBack, WallFront, WallLast
} WallId;

typedef enum FontId {
  FontHelp = 0, FontBanner, FontLast
} FontId;

typedef enum TexId {
  TexGrid = 0, TexChecker, TexPlate, TexOGL, TexPhoenix, TexFexp5, TexFrap, TexGeorge, TexLast
} TexId;

typedef enum MirrorMode {
  MirrorNone = 0, MirrorGround = 1, MirrorWall = 2, MirrorGroundWall = 3, MirrorLast = 4
} MirrorMode;

typedef enum SurfaceMode {
  SModePoint, SModeWireV, SModeWireH, SModePlain, SModeCMap
} SurfaceMode;


typedef struct Material {
  A4D ambient;
  A4D diffuse;
  A4D emission;
  A4D specular;
  A1D shininess;
} Material;

typedef struct Light {
  A3D sPos;
  A3D sSpeed;
  A4D ambient;
  A4D diffuse;
  A4D specular;
  UBool on;			/* light on? */
  Float32 rx[MirrorLast], ry[MirrorLast], rz[MirrorLast];	/* raster position */
  Float32 rs[MirrorLast];	/* raster size of light halo */
  Float32 oz[MirrorLast];	/* ortho z position of light */
  Float32 zBefore[MirrorLast];	/* z before surface draw to detect oclusion */
  UBool hidden[MirrorLast];	/* light hidden by something? */
  Float32 fade[MirrorLast];	/* fading factor */
  Float32 fadeSpeed[MirrorLast];	/* fading speed */
} Light;

typedef struct Flare {
  GLuint texI;			/* index of texture */
  Float32 pos;			/* position on flare axis (-1..1) */
  Float32 size;			/* size in pixels */
  A4D color;			/* color (precompensated) */
} Flare;

typedef struct Shine {
  GLuint texI;			/* index of texture */
  Float32 size;
  A4D color;
  UBool constant;		/* always draw it? */
} Shine;

typedef struct Observer {
  A3D sPos;			/* spherical coords */
  A3D sSpeed;			/* spherical speed */
  Float32 fov;			/* field of view */
  A3D cPos;			/* cartesian coords */
  UBool changed;		/* changed since last draw? */
} Observer;

typedef struct ClipPlane {
  A3D sPos;			/* spherical coords */
  A3D sSpeed;			/* spherical speed */
} ClipPlane;


typedef struct Parameter {
  Float32 *param;
  Float32 speedInc;
  Float32 speed;
  Float32 defVal;
  Float32 maxVal, minVal;
} Parameter;

typedef struct Surface Surface;
typedef void (*EvalFunc) (Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y, Float32 * z);

struct Surface {
  /* parameters (x,y,z)=f(u,v) */
  Float32 uMin, uMax, vMin, vMax;	/* bounds for parameters */
  Float32 uResReal, vResReal;	/* resolution */
  Int32 uRes, vRes;		/* integer value for resolutions */
  Float32 uSpeed, vSpeed;	/* speed of parameters shift */
  EvalFunc f;			/* function to plot */
  /* bounding box and normalization */
  Float32 xMin, xMax, yMin, yMax, zMin, zMax;
  Float32 xDim, yDim, zDim;
  char *name;
  Float32 bbDim;		/* the bigest boundding box dimension */
  Float32 scale;		/* uniform x,y,z scale factor for surface */
  A4D colorD;			/* default color for this surface */
  Float32 p1, p2, p3, p4, p5, p6, p7, p8, p9, p0;	/* user parameters for surfaces */
  Parameter param[MaxParameterNb];	/* parameter controls */
  Int32 paramNb;
  Int32 curPa, curPb;		/* the two current parameters (index) */
  /* evaluations and normals 
   * x,y,z,x,y,z,... for each (u,v) */
  Float32 *eval;
  Float32 *norm;
};

/* this is the bunch of ugly globals */
typedef struct globalContext {
  /* window and perspective settings */
  Int32 win;
  Int32 winOrigX, winOrigY, winWidth, winHeight;
  UBool fullScreen;
  float fov, near, far;

  /* rendering settings */
  UBool help;
  Int32 currentMode;
  SurfaceMode surfaceMode;
  MirrorMode mirrorMode;
  UBool smooth;
  UBool light;
  UBool wall;
  UBool wallLight;
  UBool proj;
  UBool clip;
  UBool fog;
  UBool fogChanged;
  UBool tri;
  UBool norm;
  UBool grid;
  UBool banner;
  UBool flare;
  UBool wallTexture;
  Float32 mirrorTransp;
  Float32 mirrorTranspInc;

  A3D backgroundColor;
  A3D fogColor;
  A4D helpColor;
  A4D gridColor;
  A4D pointColor;
  A4D wireColor;
  A4D normColor;
  A4D norm2Color;
  Int32 gridRes;
  Float32 boxSize;
  Int32 wallResolution;

  /* lights (and light model) */
  A1D lModelLocal;
  A1D lModelTwoSide;
  A4D lModelAmbient;
  Light lights[LightNb];

  /* materials */
  Material materials[MatLast];

  /* observers */
  Observer obs;

  /* cLip plane */
  ClipPlane clipPlane;

  /* surfaces */
  Surface *surface;
  Surface surfaces[MaxSurfaceNb];
  Int32 surfaceI;
  Int32 surfaceNb;

  /* stats */
  unsigned long frame;		/* frame id */
  unsigned long triangleDrawn;	/* number of polys drawn since chrono start */
  DWChrono chrono;		/* chrono for long range fps */
  DWChrono frameChronos[FrameRateCount];	/* sliding instant fps  */
  Int32 fci;			/* index of current chrono */
  Float32 spf;			/* instant 1/(frame per s) */

  /* texts */
  char statText[64];
  Int32 statTextLen;
  Float32 statScale;
  char **helpTexts;
  Int32 helpTextLen;
  Float32 helpScale;
  char *bannerText;
  Float32 *bannerOffsets;
  Int32 bannerTextLen;
  Float32 bannerScale;
  Float32 bannerOffset;
  Float32 bannerSpeed;
  UByte bannerColor1[4], bannerColor2[4];

  /* flares and shines */
  Flare flares[MaxFlareNb];
  Int32 flareNb;
  Shine shines[MaxShineNb];
  Int32 shineNb;

  /* fonts (texture fonts) */
  TexFont *fonts[FontLast];

  /* textures */
  GLuint texIds[TexLast];
  Int32 texI;			/* current texture id index */
  GLuint texFlares[TexFlareNb];
  GLuint texShines[TexShineNb + TexFlareNb];
  GLuint wallTextures[WallLast];

  /* matrices (for glu(un)project) */
  Matd44 projMat;
  Matd44 modelMat;
  Int32 viewport[4];

} globalContext;


globalContext gc;



Module char *helpTexts[] =
{
  " Usage",
  "",
  "        F1/? : Help (this screen) on/off.",
#ifdef XMESA
  "           ! : Full screen on/off (XMesa users).",
#endif
  "         +/- : next/prev predefined display modes.",
  "     alt +/- : next/prev predefined surface.",
  "           s : [S]mooth on/off.",
  "           w : [W]alls on/off.",
  "           l : [L]ighting for walls on/off.",
  "           v : [V]isible lamp on/off.",
  "           p : [P]rojection on/off.",
  "           n : [N]ormals on/off.",
  "           c : [C]lip plane on/off.",
  "           g : [G]rid on/off.",
  "           f : [F]og on/off.",
  "         m/M : [M]irror mode next/prev.",
  "         i/I : Mirror transm[i]ssion +/-.",
  "         t/T : [T]exture next/prev.",
  "       [0-9] : Set the two current parameters.",
  "shift arrows : Change current parameters values.",
  " ctrl arrows : Move clip plane in surface.",
  "  alt arrows : Move light around surface.",
  "      arrows : Move eye around surface.",
  "         d/D : [D]istance from surface +/-.",
  "         r/R : Surface [r]esolution +/-.",
  "           z : Re[Z]et all positions to default.",
  "       space : Stop all animations.",
  "           S : Screen dump (png).",
  "    F,W,a... : More options, (screen too small ;)",
  "         Esc : Quit.",
  "",
  "                                     (c) Olivier ARSAC",
  "",
  NULL
};
#define HelpTextNb ((sizeof(helpTexts)/sizeof(helpTexts[0]))-1)

Module char bannerText[] = "OpenGL rewlz!       KOala frags...     Oliv do bugs ;)     Quake shakes :)       Direct3D sux!!!      Hell to the king babe!";

Module Float32 colorMap[][4] =
{
  {0.5, 0.0, 1.0, 1.0} /* violet */ ,
  {0.0, 0.0, 1.0, 1.0} /* blue */ ,
  {0.0, 1.0, 1.0, 1.0} /* cyan */ ,
  {0.0, 1.0, 0.0, 1.0} /* green */ ,
  {1.0, 1.0, 0.0, 1.0} /* yellow */ ,
  {1.0, 0.5, 0.0, 0.9} /* orange */ ,
  {1.0, 0.0, 0.0, 0.7} /* red */ ,
  {1.0, 1.0, 1.0, 0.3} /* white */ ,
  {0.0, 0.0, 0.0, 0.0} /* black */ ,
};
#define ColorMapNb (sizeof(colorMap)/sizeof(colorMap[0]))

Float32 ColRed[4] =
{1, 0, 0, 1};
Float32 ColGreen[4] =
{0, 1, 0, 1};
Float32 ColBlue[4] =
{0, 0, 1, 1};
Float32 ColWhite[4] =
{1, 1, 1, 1};

Module Int32 predefinedModes[][17] =
{
  //  {SModePlain, 1, 1, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, CR1},
  //  {SModePlain, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, CR1},
  /* surfaceMode,smooth,wall,light,wallLight,clip,mirror,proj,norm,texId,fog,tri,grid,banner,flares,walltextures,surfaceRes */
  /* surfMode,sm,wa,li,wl,cl,mi,pr,no,ti,fo,tr,gr,ba,fl,wt,cr */
  {SModePoint, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 40},
  {SModeWireV, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 25},
  {SModePlain, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 20},
  {SModePlain, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, CR1},
  {SModePlain, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, CR1},
  {SModePlain, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, CR1},
  {SModePlain, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, CR1},
  {SModePlain, 1, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, CR1},
  {SModePlain, 1, 1, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, CR1},
  {SModePlain, 1, 1, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 1, CR1},
  {SModePlain, 1, 1, 1, 1, 1, 2, 0, 0, 1, 0, 0, 0, 1, 1, 1, CR1},
  {SModePlain, 1, 1, 1, 1, 1, 3, 0, 0, 2, 0, 0, 0, 1, 1, 1, CR1},
  {SModePlain, 1, 1, 1, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, CR1},
  {SModePlain, 1, 1, 1, 1, 0, 1, 1, 0, 0, 0, 0, 1, 0, 0, 0, CR1},
  {SModePlain, 1, 1, 1, 1, 0, 1, 1, 1, 0, 0, 0, 1, 0, 0, 0, CR1},
  {SModePlain, 1, 1, 1, 1, 0, 1, 1, 1, 2, 0, 0, 1, 0, 0, 0, CR1},
  {SModeCMap, 1, 1, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, CR1},
  {SModePlain, 1, 1, 1, 1, 1, 3, 1, 1, 3, 1, 1, 1, 1, 1, 0, CR1},
  /* surfMode,sm,wa,li,wl,cl,mi,pr,no,ti,fo,tr,gr,ba,fl,wt,cr */
};
#define ModeNb (sizeof(predefinedModes)/sizeof(predefinedModes[0]))

/* identity matrix */
Matd44 idMat =
{1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1};


/* pop the projection matrix */
#define PopProjection() \
  glMatrixMode(GL_PROJECTION);\
  glPopMatrix();\
  glMatrixMode(GL_MODELVIEW);

/* check if a mirror image has to be drawn depenseding on current mirrorMode */
#define MirrorImage(_I,_MM) ((_I <= _MM) && !((_I==MirrorGround)&&(_MM==MirrorWall)))

/* this part is automaticaly updated, do NOT edit below */
/**************************************************************** Prototypes */
static void switchLight(UBool nl);
static void makeStripeImage(UByte * texImage, Int32 w, Int32 h);
static void makeCheckImage(UByte * texImage, Int32 w, Int32 h);
static void makeCircleImage(UByte * texImage, Int32 w, Int32 h);
static void funcBessel(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 *
		       y, Float32 * z);
static void funcQuad(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y
		     ,Float32 * z);
static void funcWaves(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 *
		      y, Float32 * z);
static void funcTorus(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 *
		      y, Float32 * z);
static void funcInspiration(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y, Float32 * z);
static void funcBoy(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y,
		    Float32 * z);
static void funcMonkeySaddle(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y, Float32 * z);
static void funcShell(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 *
		      y, Float32 * z);
static void funcBohemianDome(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y, Float32 * z);
static void funcAstrodalEllipsoid(Surface * c, Float32 u, Float32 v, Float32 * x
				  ,Float32 * y, Float32 * z);
static void funcEnneper(Surface * c, Float32 u, Float32 v, Float32 * x, Float32
			* y, Float32 * z);
static void funcKleinBottle(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y, Float32 * z);
static void funcKuen(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y
		     ,Float32 * z);
static void setRasterOrthoProj(void);
static void setObs(Observer * obs);
static void setFog(void);
static Int32 setLights(MirrorMode m);
static void setMaterial(MaterialId matId, int face);
static void setClipPlane(ClipPlane * cp);
static void unsetClipPlane(ClipPlane * cp);
static void drawHelp(void);
static void drawBanner(void);
static void drawTri(void);
static void drawGrid(WallId wid);
static void drawWall(Float32 xb, Float32 yb, Float32 xt, Float32 yt, Float32 z,
		     Float32 rx, Float32 ry, Int32 res);
static void drawMirror(MirrorMode mId);
static void drawWalls(void);
static void drawClipPlane(Surface * c, ClipPlane * cp);
static UBool hideLight(Light * light, UBool hide, MirrorMode m);
static void pickZLights(MirrorMode m);
static void checkZLights(MirrorMode m);
static void drawLights(MirrorMode m);
static void drawFlares(Light * light, Float32 transp, MirrorMode m);
static void drawSurfaceNorm(Surface * c);
static void drawSurfaceProjPoint(Surface * c);
static void drawSurfaceProjWire(Surface * c);
static void drawSurfaceProjPlain(Surface * c);
static void drawSurfaceProjTexture(Surface * c);
static void drawSurfaceProj(Surface * c);
static void drawSurfacePoint(Surface * c);
static void drawSurfaceWireV(Surface * c);
/* wire lines but with hiden lines removal
 * brute force o:)
 * draw a transparent surface then decal and redraw line with Z test against pre
 set surface
 * should be optimized
 */
static void drawSurfaceWireH(Surface * c);
static void drawSurfacePlain(Surface * c);
static void drawSurfaceCMap(Surface * c);
static void drawSurfaceTexture(Surface * c);
static void drawSurface(Surface * c);
static void drawScene(void);
static void updateFrameRate(void);
static void updateObs(Observer * obs);
static void updateClipPlane(ClipPlane * cp);
static void updateLight(Light * l);
static void updateMirror(void);
static Int32 updateParam(Parameter * p);
static void updateSurface(Surface * c);
static void updateBanner(void);
static void updateScene(void);
static void reshapeWindow(int width, int height);
static void speedObs(Observer * obs, Float32 rs, Float32 ts, Float32 ps);
static void speedParameter(Parameter * p, Float32 pis);
static void speedLight(Light * l, Float32 ts, Float32 ps);
static void speedPlane(ClipPlane * cp, Float32 rs, Float32 ts);
static void specialEventCB(int key, int x, int y);
static void setPredefinedMode(int i);
static void keyEventCB(unsigned char key, int x, int y);
static void fillSurface(Surface * c);
static void reinitSurface(Surface * c);
static void initParam(Surface * c, Float32 * param, Float32 speedInc, Float32 fVal, Float32 minVal, Float32 maxVal);
static void initSurface(Surface * c);
static void initSurfaces(void);
static Observer *initObserver(Observer * obs);
static void initLights(void);
static void initMaterials(void);
static void reinitPos(void);
static void loadFont(TexFont ** ptxf, char *fileName);
static void initFonts(void);
static void initFlares(void);
static void initTextures(void);
static void initGc(int *argc, char **argv);
/* OpenGL settings 
 */
static void initOGL(void);


/************************************************************ End Prototypes */
/* end of automaticaly updated part */

Module void
makeStripeImage(UByte * texImage, Int32 w, Int32 h)
{
  Int32 i, j;
  UByte c1, c, *p;

  p = texImage;
  for (i = 0; i < w; i++) {
    c1 = (i % (w >> 3)) ? 0xff : 0;
    for (j = 0; j < h; j++) {
      c = c1 & ((j % (h >> 3)) ? 0xff : 0);
      *p++ = c;
      *p++ = c;
      *p++ = c;
      *p++ = c;
    }
  }
}

Module void
makeCheckImage(UByte * texImage, Int32 w, Int32 h)
{
  Int32 i, j;
  UByte c1, c, *p;

  p = texImage;

  for (i = 0; i < w; i++) {
    c1 = ((i / (w >> 3)) & 0x01) ? 0xff : 0;
    for (j = 0; j < h; j++) {
      c = c1 ^ (((j / (h >> 3)) & 0x01) ? 0xff : 0);
      *p++ = c;
      *p++ = c;
      *p++ = c;
      *p++ = c;
    }
  }
}

Module void
makeCircleImage(UByte * texImage, Int32 w, Int32 h)
{
  Int32 i, j, s;
  Float32 x, y;
  UByte c, *p;

  p = texImage;
  s = w >> 2;
  for (i = 0; i < w; i++) {
    x = i % s;
    for (j = 0; j < h; j++) {
      y = j % s;
      c = 255 * (x * x + y * y) / (s * s);
      *p++ = c;
      *p++ = c;
      *p++ = c;
      *p++ = c;
    }
  }
}


Module void
funcBessel(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y, Float32 * z)
{
  double sd, d;
  *x = u;
  *y = v;
  u += c->p1;
  v += c->p2;
  sd = (u * u) + (v * v);
  d = sqrt(sd);
  *z = cos(d) + c->p3 * sin(u) + c->p4 * sin(v);
}

Module void
funcQuad(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y, Float32 * z)
{
  *x = u;
  *y = v;
  u += c->p1;
  v += c->p2;
  *z = u * u - v * v + u * c->p3 + v * c->p4;
}

Module void
funcWaves(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y, Float32 * z)
{
  *x = u;
  *y = v;
  u += c->p1;
  v += c->p2;
  *z = sin(c->p3 * u) * cos(c->p4 * v);
}

Module void
funcTorus(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y, Float32 * z)
{
  Float32 cv;

  u = u * c->p5;
  v = v * c->p6;
  cv = c->p1 * cos(v);
  *x = c->p3 * cos(u) * (c->p2 + cv);
  *y = sin(u) * (c->p2 + cv);
  *z = c->p1 * sin(c->p4 * v);
}

Module void
funcInspiration(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y, Float32 * z)
{
  Float32 cv;

  cv = c->p1 * cos(v);
  *x = cos(c->p4 * u) * (1 + cv);
  *y = sin(c->p4 * u) * (1 + cv);
  *z = c->p1 * sin(v) + c->p2 * cos(c->p3 * u);
}

Module void
funcBoy(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y, Float32 * z)
{
  Float32 A, A2, B, B2, a, x1, z1, sina, cosu, sinu, cosv, sinv;

  u = u * c->p2;
  v = v * c->p3;
  cosu = cos(u);
  sinu = sin(u);
  cosv = cos(v);
  sinv = sin(v);
  A = 10 + 1.41 * sin(6 * u - Pi / 3) + 1.98 * sin(3 * u - Pi / 6);
  B = 10 + 1.41 * sin(6 * u - Pi / 3) - 1.98 * sin(3 * u - Pi / 6);
  A2 = A * A;
  B2 = B * B;
  a = Pi / 8 * sin(3 * c->p1 * u);
  x1 = (A2 - B2) / sqrt(A2 + B2) + A * cosv - B * sinv;
  z1 = sqrt(A2 + B2) + A * cosv + B * sinv;
  sina = sin(a);
  *x = x1 * cosu - z1 * sina * sinu;
  *y = x1 * sinu + z1 * sina * cosu;
  *z = z1 * cos(a);
}

Module void
funcMonkeySaddle(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y, Float32 * z)
{
  u = u * c->p2;
  v = v * c->p3;
  *x = u * sin(v);
  *y = u * cos(v);
  *z = -u * u * u * sin(c->p1 * v);
}

Module void
funcShell(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y, Float32 * z)
{
  Float32 sinu;

  u = u * c->p4;
  v = v * c->p5;
  sinu = sin(u);
  *x = -c->p2 * v * sinu * cos(c->p3 * v);
  *y = -c->p2 * v * sinu * sin(c->p3 * v);
  *z = c->p2 * v * cos(u) + c->p1 * v;
}

Module void
funcBohemianDome(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y, Float32 * z)
{
  u = u * c->p4;
  v = v * c->p5;
  *x = c->p1 * cos(u);
  *y = c->p2 * cos(v) + c->p1 * sin(u);
  *z = c->p3 * sin(v);
}

Module void
funcAstrodalEllipsoid(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y, Float32 * z)
{
  Float32 cosv;

  u = u * c->p4;
  v = v * c->p5;
  cosv = cos(v);
  *x = pow(c->p1 * cos(u) * cosv, 3);
  *y = -pow(c->p2 * sin(u) * cosv, 3);
  *z = pow(c->p3 * sin(v), 3);
}

Module void
funcEnneper(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y, Float32 * z)
{
  *x = c->p2 * u - (pow(u, 3) / 3) + c->p1 * u * v * v;
  *y = c->p2 * v - (pow(v, 3) / 3) + c->p1 * u * u * v;
  *z = u * u - v * v;
}

Module void
funcKleinBottle(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y, Float32 * z)
{
  Float32 cosu, sinu, cosv, sinv, cosu2, sinu2;

  u = u * c->p1;
  v = v * c->p2;
  cosu = cos(u);
  cosu2 = cos(u * .5);
  sinu2 = sin(u * .5);
  sinu = sin(u);
  cosv = cos(v);
  sinv = sin(v);
  *x = cosu * (cosu2 * (Sqrt2 + cosv) + (sinu2 * sinv * cosv));
  *y = sin(u) * (cosu2 * (Sqrt2 + cosv) + (sinu2 * sinv * cosv));
  *z = -1 * sinu2 * (Sqrt2 + cosv) + cosu2 * sinv * cosv;
}

Module void
funcKuen(Surface * c, Float32 u, Float32 v, Float32 * x, Float32 * y, Float32 * z)
{
  Float32 cosu, sinu, cosv, sinv;

  u = u * c->p1;
  v = v * c->p2;
  cosu = cos(u);
  sinu = sin(u);
  cosv = cos(v);
  sinv = sin(v);
  *x = 2 * (cosu + u * sinu) * sinv / (1 + u * u * sinv * sinv);
  *y = 2 * (sinu - u * cosu) * sinv / (1 + u * u * sinv * sinv);
  *z = log(tan(v / 2)) + 2 * cosv / (1 + u * u * sinv * sinv);
}

World void
showZBuffer(void)
{
  UByte *pixels;

  pixels = UMallocZ(gc.winWidth * gc.winHeight);
  glFinish();
  glReadBuffer(GL_BACK);
  glReadPixels(0, 0, gc.winWidth, gc.winHeight, GL_DEPTH_COMPONENT, GL_UNSIGNED_BYTE, pixels);
  glDrawBuffer(GL_FRONT);
  glRasterPos2i(0, 0);
  glDrawPixels(gc.winWidth, gc.winHeight, GL_RED, GL_UNSIGNED_BYTE, pixels);
  glDrawPixels(gc.winWidth, gc.winHeight, GL_GREEN, GL_UNSIGNED_BYTE, pixels);
  glDrawPixels(gc.winWidth, gc.winHeight, GL_BLUE, GL_UNSIGNED_BYTE, pixels);
  glDrawBuffer(GL_BACK);
  UFree(pixels);
}

World void
showBBuffer(void)
{
  UByte *pixels;

  pixels = UMallocZ(3 * gc.winWidth * gc.winHeight);
  glFinish();
  glReadBuffer(GL_BACK);
  glReadPixels(0, 0, gc.winWidth, gc.winHeight, GL_RGB, GL_UNSIGNED_BYTE, pixels);
  glDrawBuffer(GL_FRONT);
  glRasterPos2i(0, 0);
  glDrawPixels(gc.winWidth, gc.winHeight, GL_RGB, GL_UNSIGNED_BYTE, pixels);
  glDrawBuffer(GL_BACK);
  UFree(pixels);
}

Module void
setRasterOrthoProj(void)
{
  // set orthographic projection to match coords/pixel adress
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  glOrtho(-.5, gc.winWidth - .5, -.5, gc.winHeight - .5, 0, gc.winWidth);
  glMatrixMode(GL_MODELVIEW);
}

Module void
setObs(Observer * obs)
{
  Float32 dz;

  if (obs->changed) {
    dz = gc.boxSize / 10;
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(obs->fov, gc.winWidth / (float) gc.winHeight, gc.near, gc.far);
    SpherToCart(obs->sPos, obs->cPos);
    Z(obs->cPos) += dz;
    gluLookAt(A3DGet(obs->cPos), 0, 0, dz, 0, 0, 1);
    glGetDoublev(GL_PROJECTION_MATRIX, gc.projMat);
    glMatrixMode(GL_MODELVIEW);
    obs->changed = UFalse;
  }
}

Module void
setFog(void)
{
  if (gc.fogChanged) {
    if (gc.fog) {
      glEnable(GL_FOG);
      glFogi(GL_FOG_MODE, GL_EXP);
      glFogfv(GL_FOG_COLOR, gc.fogColor);
      glFogf(GL_FOG_DENSITY, 0.25);
      glHint(GL_FOG_HINT, GL_NICEST);
    }
    else {
      glDisable(GL_FOG);
    }
    gc.fogChanged = UFalse;
  }
}

Module Int32
setLights(MirrorMode m)
{
  Matd44 modelMat, projMat;
  Float64 rx, ry, rz, rx2, ry2, rz2, dummy, oz;
  Float32 dx, dy, dz, t;
  Light *light;
  A4D cPos;
  Observer *obs;
  A3D v;
  Int32 i, li, lo = 0;

  li = GL_LIGHT0;
  for (i = 0; i < LightNb; i++) {
    light = &(gc.lights[i]);
    if (light->on) {
      lo++;
      SpherToCart(light->sPos, cPos);
      // 0-> directional
      // 1-> positional
      W(cPos) = 1;

      glLightfv(li + i, GL_POSITION, cPos);
      glLightfv(li + i, GL_AMBIENT, light->ambient);
      glLightfv(li + i, GL_DIFFUSE, light->diffuse);
      glLightfv(li + i, GL_SPECULAR, light->specular);
      glEnable(li + i);

      /* set all fields used to draw light */
      if (!gc.light) {
	/* this light is useless (completly fadded out)
	 * thus abort costly computations */
	continue;
      }

      /* assume lamp is visible then do vivibility checks */
      hideLight(light, UFalse, m);

      /* lamp hidden by ground or mirror */
      if ((gc.wall || (gc.mirrorMode & MirrorGround)) && (Z(cPos) < 0)) {
	if (hideLight(light, UTrue, m))
	  continue;
      }

      /* find raster position of light */
      glGetDoublev(GL_MODELVIEW_MATRIX, modelMat);
      gluProject(A3DGet(cPos), modelMat, gc.projMat, gc.viewport, &rx, &ry, &rz);

      /* lamp out of view */
      if ((rx < 0) || (rx > gc.winWidth) || (ry < 0) || (ry > gc.winHeight) || (rz < 0) || (rz > 1)) {
	if (hideLight(light, UTrue, m))
	  continue;
      }

      /* find raster size of light (to match perspective size change) */
      /* FIXME do real bilboard here (for mirror lights) */
      obs = &gc.obs;
      t = Y(obs->sPos) + Pi_2;
      A3DSet(v, cos(t), sin(t), 0);
      A3DMult(v, gc.boxSize * .005);
      A3DAddA3D(cPos, v);
      gluProject(A3DGet(cPos), modelMat, gc.projMat, gc.viewport, &rx2, &ry2, &rz2);
      dx = (rx2 - rx);
      dy = (ry2 - ry);
      dz = (rz2 - rz);
      light->rs[m] = sqrt(dx * dx + dy * dy + dz * dz);

      /* find z to use in ortho mode to match z of light in perspective mode */
      setRasterOrthoProj();
      glGetDoublev(GL_PROJECTION_MATRIX, projMat);
      gluUnProject(rx, ry, rz, idMat, projMat, gc.viewport, &dummy, &dummy, &oz);
      light->rx[m] = rx;
      light->ry[m] = ry;
      light->rz[m] = rz;
      light->oz[m] = oz;
      PopProjection();
    }
    else {
      glDisable(li + i);
      continue;
    }
  }
  return lo;
}

Module void
setMaterial(MaterialId matId, int face)
{
  Material *mat;

  glEnable(GL_LIGHTING);
  mat = &(gc.materials[matId]);
  glMaterialfv(face, GL_AMBIENT, mat->ambient);
  glMaterialfv(face, GL_DIFFUSE, mat->diffuse);
  glMaterialfv(face, GL_EMISSION, mat->emission);
  glMaterialfv(face, GL_SPECULAR, mat->specular);
  glMaterialfv(face, GL_SHININESS, mat->shininess);
}

Module void
setClipPlane(ClipPlane * cp)
{
  Float64 eq[4];

  if (gc.clip) {
    glPushMatrix();
    eq[0] = cos(cp->sPos[1]);
    eq[1] = sin(cp->sPos[1]);
    eq[2] = 0;
    eq[3] = cp->sPos[0];
    glClipPlane(GL_CLIP_PLANE0, eq);
    glEnable(GL_CLIP_PLANE0);
    glPopMatrix();
  }
}

Module void
unsetClipPlane(ClipPlane * cp)
{
  glDisable(GL_CLIP_PLANE0);
}

Module void
drawHelp(void)
{
  Int32 w, h, cx, cy, dy, dx;
  Float32 is;
  unsigned long ms;
  char **text;

  w = gc.winWidth;
  h = gc.winHeight;

  glPushMatrix();

  setRasterOrthoProj();

  // fps and pps stats
  if ((gc.frame <= FrameStats) || !(gc.frame % FrameStats)) {
    ms = DWChronoGet(&gc.chrono);
    ms = Max(ms, 1);
    sprintf(gc.statText, "%5.2fF/s %7dP/s   %s  (%d params: %d %d)", (FrameStats * 1000.) / ms, (int) ((1000 * gc.triangleDrawn) / ms), (gc.surface->name) ? gc.surface->name : "unknown", gc.surface->paramNb, gc.surface->curPa + 1, gc.surface->curPb + 1);
    gc.statTextLen = strlen(gc.statText);
  }

  glDisable(GL_LIGHTING);
  glDepthFunc(GL_ALWAYS);
  glColor4fv(gc.helpColor);
  txfBindFontTexture(gc.fonts[FontHelp]);

  /* stat text */
  glLoadIdentity();
  glTranslatef(10, 10, 0);
  glScalef(gc.statScale, gc.statScale, 1);
  txfRenderString(gc.fonts[FontHelp], gc.statText);
  gc.triangleDrawn += 2 * gc.statTextLen;

  /* help text */
  if (gc.help) {
    glLoadIdentity();

    // big square 
    glColor4f(.0, .0, .0, .5);
    glDisable(GL_TEXTURE_2D);
    glRectf(w * HelpMargin, h * HelpMargin,
	    w * (1 - HelpMargin), h * (1 - HelpMargin));
    glEnable(GL_TEXTURE_2D);
    glColor4fv(gc.helpColor);
    is = 1 / gc.helpScale;

    // set constants
    cx = (w * 2 * HelpMargin);
    dx = 0;
    dy = (-h * (1 - 2 * HelpMargin) / (HelpTextNb));
    cy = (h * (1 - HelpMargin) + dy);
    dy *= is;
    // draw all strings
    glTranslatef(cx, cy, 0);
    glScalef(gc.helpScale, gc.helpScale, 1);
    text = gc.helpTexts;
    while (*text) {
      glTranslatef(dx, dy, 0);
      dx = -txfRenderString(gc.fonts[FontHelp], *text);
      text++;
    }
    gc.triangleDrawn += 2 * gc.helpTextLen;
  }
  glDepthFunc(GL_LEQUAL);
  glDisable(GL_TEXTURE_2D);
  glPopMatrix();
  PopProjection();
}

Module void
drawBanner(void)
{
  Int32 w, a, d, head, i;
  Float32 s, dx, endx, wallOffset, bannerOffset, z;
  char *string;

  if (!gc.banner)
    return;

  txfGetStringMetrics(gc.fonts[FontBanner], "O", 1, &w, &a, &d);
  UAssert(w != 0);
  s = 2 * gc.boxSize * gc.bannerScale / w;
  wallOffset = w / gc.bannerScale;
  txfBindFontTexture(gc.fonts[FontBanner]);
  setMaterial(MatBanner, GL_FRONT_AND_BACK);
  //  glEnable(GL_COLOR_MATERIAL);
  glShadeModel(GL_SMOOTH);
  /* that removes flickering on overlapping banner chars...
   * but this seems to slow mesa3/v2 down (to knee down)
   glColorMask(GL_TRUE,GL_TRUE,GL_TRUE, GL_FALSE);
   */

  z = gc.boxSize / 3;
  /* first wall (right) */
  bannerOffset = gc.bannerOffset;
  head = -1;
  for (i = 0; i < gc.bannerTextLen;) {
    if (bannerOffset <= gc.bannerOffsets[++i]) {
      head = i - 1;
      break;
    }
  }
  if (head < 0) {
    head = 0;
    bannerOffset = gc.bannerOffset = -wallOffset * 3;
  }
  glPushMatrix();
  glTranslatef(gc.boxSize * .99, gc.boxSize - s * (gc.bannerOffsets[head] - bannerOffset), z);
  glRotatef(-90, 0, 1, 0);
  glRotatef(-90, 0, 0, 1);
  glScalef(s, s, s);
  i = 0;
  string = gc.bannerText + head;
  dx = gc.bannerOffsets[head];
  endx = wallOffset + bannerOffset;
  while ((dx <= endx) && (*string)) {
    if ((*string) == 'O') {
      dx += txfRenderGlyph(gc.fonts[FontBanner], *(string++), bannerOffset);
    }
    else {
      dx += txfRenderGlyph(gc.fonts[FontBanner], *(string++), 0);
      //    dx += txfRenderGlyphSmooth(gc.fonts[FontBanner], *(string++), gc.bannerColor1, gc.bannerColor2);
    }
  }
  glPopMatrix();
  glPushMatrix();
  /* second wall (back) */
  bannerOffset += wallOffset;
  head = -1;
  for (i = 0; i < gc.bannerTextLen;) {
    if (bannerOffset <= gc.bannerOffsets[++i]) {
      head = i - 1;
      break;
    }
  }
  if (head >= 0) {
    glTranslatef(gc.boxSize - s * (gc.bannerOffsets[head] - bannerOffset), -gc.boxSize * .99, z);
    glRotatef(-90, 1, 0, 0);
    glRotatef(180, 0, 0, 1);
    glScalef(s, s, s);
    i = 0;
    string = gc.bannerText + head;
    dx = gc.bannerOffsets[head];
    endx = wallOffset + bannerOffset;
    while ((dx <= endx) && (*string)) {
      if ((*string) == 'O') {
	dx += txfRenderGlyph(gc.fonts[FontBanner], *(string++), bannerOffset);
      }
      else {
	dx += txfRenderGlyph(gc.fonts[FontBanner], *(string++), 0);
      }
    }
  }
  glPopMatrix();
  glPushMatrix();
  /* third wall (left) */
  bannerOffset += wallOffset;
  head = -1;
  for (i = 0; i < gc.bannerTextLen;) {
    if (bannerOffset <= gc.bannerOffsets[++i]) {
      head = i - 1;
      break;
    }
  }
  if (head >= 0) {
    glTranslatef(-gc.boxSize * .99, -gc.boxSize + s * (gc.bannerOffsets[head] - bannerOffset), gc.boxSize / 3);
    glRotatef(90, 0, 1, 0);
    glRotatef(90, 0, 0, 1);
    glScalef(s, s, s);
    i = 0;
    string = gc.bannerText + head;
    dx = gc.bannerOffsets[head];
    endx = wallOffset + bannerOffset;
    while ((dx <= endx) && (*string)) {
      if ((*string) == 'O') {
	dx += txfRenderGlyph(gc.fonts[FontBanner], *(string++), bannerOffset);
      }
      else {
	dx += txfRenderGlyph(gc.fonts[FontBanner], *(string++), 0);
      }
    }
  }
  glPopMatrix();
  gc.triangleDrawn += 2 * gc.bannerTextLen;
  glDisable(GL_COLOR_MATERIAL);
  glDisable(GL_TEXTURE_2D);
  /* see above comment on glColorMask
     glColorMask(GL_TRUE,GL_TRUE,GL_TRUE, GL_TRUE); */
}

Module void
drawTri(void)
{
  Float32 dz;

  if (!gc.tri)
    return;

  dz = gc.boxSize / 90;
  glDisable(GL_LIGHTING);
  glEnable(GL_LINE_SMOOTH);
  glBegin(GL_LINES);
  glColor4f(1, 0, 0, 1);
  glVertex3f(0, 0, dz);
  glVertex3f(1, 0, dz);
  glColor4f(0, 1, 0, 1);
  glVertex3f(0, 0, dz);
  glVertex3f(0, 1, dz);
  glColor4f(0, 0, 1, 1);
  glVertex3f(0, 0, dz);
  glVertex3f(0, 0, 1);
  glEnd();
  glDisable(GL_LINE_SMOOTH);
}

Module void
drawGrid(WallId wid)
{
  Surface *c;
  A3D a, b;
  Float32 incx, incy, incz, scale, xMin, xMax, yMin, yMax, zMin, zMax,
    dz, eps, p, p2;

  if (!gc.grid)
    return;

  glEnable(GL_LINE_SMOOTH);
  glDisable(GL_LIGHTING);
  glColor4fv(gc.gridColor);
  c = gc.surface;
  scale = c->scale;
  eps = 0.000001 / gc.gridRes;
  dz = gc.boxSize / 20;
  p = log10(Max(c->xDim, 0.0000001));
  p2 = log10(Max(c->yDim, 0.0000001));
  p = Max(p2, p);
  p2 = log10(Max(c->zDim, 0.0000001));
  p = Max(p2, p);
  p = pow(10, rint(p)) / gc.gridRes;

  incy = p;
  yMin = scale * incy * floor(c->yMin / incy);
  yMax = scale * incy * ceil(c->yMax / incy);
  incy *= scale;
  incy = Max(incy, eps);
  yMax = Max(yMin + incy, yMax);

  glLineWidth(1);
  glBegin(GL_LINES);

  if (wid == WallGround) {
    // ground grid
    incx = p;
    xMin = scale * incx * floor(c->xMin / incx);
    xMax = scale * incx * ceil(c->xMax / incx);
    incx *= scale;
    incx = Max(incx, eps);	/* infinite loop if incx==0 */
    xMax = Max(xMin + incx, xMax);	/* avoid xMin==xMax */
    Z(a) = Z(b) = gc.boxSize / 100;
    X(a) = xMin;
    X(b) = xMin;
    Y(a) = yMin;
    Y(b) = yMax;
    while (X(b) <= xMax + eps) {
      glVertex3fv(a);
      glVertex3fv(b);
      X(a) += incx;
      X(b) += incx;
    }
    X(a) = xMin;
    X(b) = xMax;
    Y(a) = yMin;
    Y(b) = yMin;
    while (Y(b) <= yMax + eps) {
      glVertex3fv(a);
      glVertex3fv(b);
      Y(a) += incy;
      Y(b) += incy;
    }
  }
  else if (wid == WallRight && gc.proj) {
    // wall grid
    incz = p;
    zMin = dz + scale * (incz * floor(c->zMin / incz) - c->zMin);
    zMax = dz + scale * (incz * ceil(c->zMax / incz) - c->zMin);
    incz *= scale;
    incz = Max(incz, eps);
    zMax = Max(zMin + incz, zMax);

    X(a) = X(b) = gc.boxSize * .995;
    Z(a) = zMin;
    Z(b) = zMax;
    Y(a) = yMin;
    Y(b) = yMin;
    while (Y(b) <= yMax + eps) {
      glVertex3fv(a);
      glVertex3fv(b);
      Y(a) += incy;
      Y(b) += incy;
    }
    Z(a) = zMin;
    Z(b) = zMin;
    Y(a) = yMin;
    Y(b) = yMax;
    while (Z(b) <= zMax + eps) {
      glVertex3fv(a);
      glVertex3fv(b);
      Z(a) += incz;
      Z(b) += incz;
    }
  }
  glEnd();
  glDisable(GL_LINE_SMOOTH);
  glDisable(GL_COLOR_MATERIAL);
}

Module void
drawWall(Float32 xb, Float32 yb, Float32 xt, Float32 yt, Float32 z, Float32 rx, Float32 ry, Int32 res)
{
  Int32 i, j;
  Float32 xi, yi, x1, x2, y;

  glPushMatrix();
  glRotatef(rx, 1, 0, 0);
  glRotatef(ry, 0, 1, 0);
  glNormal3f(0, 0, -1);
  xi = (xt - xb) / res;
  yi = (yt - yb) / res;
  x1 = xb;
  x2 = x1 + xi;
  for (i = 0; i < res; i++) {
    glBegin(GL_QUAD_STRIP);
    y = yb;
    for (j = 0; j <= res; j++) {
      glVertex3f(x1, y, z);
      glVertex3f(x2, y, z);
      y += yi;
    }
    glEnd();
    x1 = x2;
    x2 += xi;
  }
  glPopMatrix();
  gc.triangleDrawn += 2 * res * res;
  // *2 becoz' each quad is tesselated into two tri bedore rasterisation
}

Module void
drawWallTexture(Float32 xb, Float32 yb, Float32 xt, Float32 yt, Float32 z, Float32 rx, Float32 ry, Int32 res)
{
  Int32 i, j;
  Float32 x1, x2, xi, y, yi, s1, s2, si, t, ti, ires;

  glPushMatrix();
  glRotatef(rx, 1, 0, 0);
  glRotatef(ry, 0, 1, 0);
  glNormal3f(0, 0, -1);
  ires = 1.f / (float) res;
  xi = (xt - xb) * ires;
  yi = (yt - yb) * ires;
  xi = (xt - xb) * ires;
  yi = (yt - yb) * ires;
  x1 = xb;
  x2 = x1 + xi;
  ti = 1 * ires;
  si = ti;
  s1 = 0;
  s2 = si;
  for (i = 0; i < res; i++) {
    glBegin(GL_QUAD_STRIP);
    y = yb;
    t = 0;
    for (j = 0; j <= res; j++) {
      glTexCoord2f(t, s1);
      glVertex3f(x1, y, z);
      glTexCoord2f(t, s2);
      glVertex3f(x2, y, z);
      y += yi;
      t += ti;
    }
    glEnd();
    x1 = x2;
    x2 += xi;
    s1 = s2;
    s2 += si;
  }
  glPopMatrix();
  gc.triangleDrawn += 2 * res * res;
}

Module void
drawMirror(MirrorMode mId)
{
  Float32 bs;
  Int32 wr;

  if (gc.wallLight)
    wr = gc.wallResolution;
  else
    wr = 1;

  bs = gc.boxSize;
  setMaterial(MatMirror, GL_FRONT_AND_BACK);
  if ((mId == MirrorGround) && (gc.mirrorMode & MirrorGround)) {
    drawWall(-bs, -bs, bs, bs, 0, 180, 0, wr);
  }
  else if ((mId == MirrorWall) && (gc.mirrorMode & MirrorWall)) {
    drawWall(-bs, -bs, bs, 0, bs, -90, 0, wr);
  }
}

Module void
drawWalls(void)
{
  Float32 bs;
  Int32 wr;

  if (!gc.wall)
    return;

  glDepthFunc(GL_ALWAYS);
  if (gc.wallLight)
    wr = gc.wallResolution;
  else
    wr = 1;

  bs = gc.boxSize;

  setMaterial(MatWall, GL_FRONT_AND_BACK);

  // ceiling
  if (gc.wallTexture && gc.wallTextures[WallTop]) {
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, gc.wallTextures[WallTop]);
    drawWallTexture(-bs, -bs, bs, bs, bs, 0, 0, wr);
    glDisable(GL_TEXTURE_2D);
  }
  else {
    drawWall(-bs, -bs, bs, bs, bs, 0, 0, wr);
  }

  // 3 plain walls
  if (gc.wallTexture && gc.wallTextures[WallBack]) {
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, gc.wallTextures[WallBack]);
    drawWallTexture(-bs, 0, bs, bs, bs, 90, 0, wr);
    glDisable(GL_TEXTURE_2D);
  }
  else {
    drawWall(-bs, 0, bs, bs, bs, 90, 0, wr);
  }

  if (gc.wallTexture && gc.wallTextures[WallRight]) {
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, gc.wallTextures[WallRight]);
    drawWallTexture(-bs, -bs, 0, bs, bs, 0, 90, wr);
    glDisable(GL_TEXTURE_2D);
  }
  else {
    drawWall(-bs, -bs, 0, bs, bs, 0, 90, wr);
  }
  if (gc.wallTexture && gc.wallTextures[WallLeft]) {
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, gc.wallTextures[WallLeft]);
    drawWallTexture(0, -bs, bs, bs, bs, 0, -90, wr);
    glDisable(GL_TEXTURE_2D);
  }
  else {
    drawWall(0, -bs, bs, bs, bs, 0, -90, wr);
  }

  // ground and missing wall if not in mirror mode
  if ((gc.mirrorTransp == 0) || !(gc.mirrorMode & MirrorGround))
    drawWall(-bs, -bs, bs, bs, 0, 180, 0, wr);
  if ((gc.mirrorTransp == 0) || !(gc.mirrorMode & MirrorWall))
    drawWall(-bs, -bs, bs, 0, bs, -90, 0, wr);

  glDepthFunc(GL_LEQUAL);
}

Module void
drawClipPlane(Surface * c, ClipPlane * cp)
{
  Float64 r, t, st, ct, xTop, xBot, yLeft, yRight, xd, yd;
  Float32 coords[8], *p, dh;	// 4 is enough in theory but numerical error may break this geometric law thus 9 is safe :)

  Int32 i, j;
  A3D p1, p2;
  A2D dxy;

  if (!gc.clip)
    return;

  glPushMatrix();
  glTranslatef(0, 0, gc.boxSize / 20);
  dh = c->zDim / (c->bbDim * ClipPlaneResolution);
  xd = yd = .5;
  r = -cp->sPos[0];		// '-' is an ugly kludge to compensate an error in equation resolution ;)

  t = cp->sPos[1];
  st = sin(t);
  ct = cos(t);
  // compute intersection with rectangle edges (-xd,-yd) (xd,yd)
  xTop = (r - yd * st) / ct;
  xBot = (r + yd * st) / ct;
  yLeft = (r + xd * ct) / st;
  yRight = (r - xd * ct) / st;

  // add all legal points to array of coords
  p = coords;
  if (Abs(yLeft) < yd) {
    *p++ = -xd;
    *p++ = yLeft;
  }
  if (Abs(yRight) < yd) {
    *p++ = xd;
    *p++ = yRight;
  }
  if (Abs(xTop) < xd) {
    *p++ = xTop;
    *p++ = yd;
  }
  if (Abs(xBot) < xd) {
    *p++ = xBot;
    *p++ = -yd;
  }

  // plot the box cliped cliping plane :)
  if ((p - coords) == 4) {
    // ok there are two legal points (not a degenerate case)
    glShadeModel(GL_SMOOTH);
    glEnable(GL_LIGHTING);
    setMaterial(MatClipPlane, GL_FRONT_AND_BACK);
    glNormal3f(-ct, -st, 0);

    A2DSet(dxy,
	   (coords[2] - coords[0]) / (ClipPlaneResolution - 1),
	   (coords[3] - coords[1]) / (ClipPlaneResolution - 1));
    for (i = 0; i < ClipPlaneResolution - 1; i++) {
      A3DSet(p1, coords[0], coords[1], i * dh);
      A3DSet(p2, coords[0], coords[1], (i + 1) * dh);
      for (j = 0; j < ClipPlaneResolution; j++) {
	glBegin(GL_QUAD_STRIP);
	glVertex3fv(p1);
	glVertex3fv(p2);
	A2DAddA2D(p1, dxy);
	A2DAddA2D(p2, dxy);
      }
      glEnd();
    }
    gc.triangleDrawn += ClipPlaneResolution * ClipPlaneResolution * 2;
    if (!gc.smooth)
      glShadeModel(GL_FLAT);

  }
  glPopMatrix();
}

Module UBool
hideLight(Light * light, UBool hide, MirrorMode m)
{
  if ((light->hidden[m] = hide)) {
    light->fadeSpeed[m] = LightFadeSpeedInc;
  }
  else {
    light->fadeSpeed[m] = -LightFadeSpeedInc;
  }
  return (light->fade[m] >= 1);
}

Module void
pickZLights(MirrorMode m)
{
  Int32 i;
  Light *light;

  if (!gc.light || !MirrorImage(m, gc.mirrorMode))
    return;

  for (i = 0; i < LightNb; i++) {
    light = &(gc.lights[i]);
    /* store current Zbuffer value at light center
     * to detect surface ocluion of ligt */
    glFinish();
    glReadPixels(light->rx[m], light->ry[m], 1, 1, GL_DEPTH_COMPONENT, GL_FLOAT, &(light->zBefore[m]));
  }
}

Module void
checkZLights(MirrorMode m)
{
  Light *light;
  Int32 i;
  Float32 zAfter;

  if (!gc.light || !MirrorImage(m, gc.mirrorMode))
    return;

  glFinish();
  for (i = 0; i < LightNb; i++) {
    light = &(gc.lights[i]);
    glReadPixels(light->rx[m], light->ry[m], 1, 1, GL_DEPTH_COMPONENT, GL_FLOAT, &zAfter);
    if ((zAfter != light->zBefore[m]) && (zAfter < light->rz[m])) {
      hideLight(light, UTrue, m);
    }
  }
}

Module void
drawLights(MirrorMode m)
{
  Float32 s, hs;
  Light *light;
  Shine *sh;
  A3D color;
  Int32 i, r, n;

  if (!gc.light)
    return;
  glDisable(GL_LIGHTING);
  glEnable(GL_TEXTURE_2D);
  glEnable(GL_BLEND);
  glBlendFunc(GL_ONE, GL_ONE);
  for (i = 0; i < LightNb; i++) {
    light = &(gc.lights[i]);
    if (light->on && (light->fade[m] < 1)) {
      setRasterOrthoProj();
      r = rand();
      n = 0;
      for (i = 0; i < gc.shineNb; i++) {
	sh = &(gc.shines[i]);
	if (!sh->constant) {
	  r = r >> 1;
	  if (r & 1)
	    continue;
	  n++;
	  if (n == 3)
	    break;
	}
	s = sh->size * light->rs[m];
	hs = s * .5;

	glPushMatrix();
	glLoadIdentity();
	glBindTexture(GL_TEXTURE_2D, sh->texI);
	A3DCp(sh->color, color);
	A3DMult(color, 1 - light->fade[m]);
	glColor3fv(color);
	glTranslatef(light->rx[m] - hs, light->ry[m] - hs, light->oz[m]);
	glBegin(GL_QUADS);
	glTexCoord2f(0, 0);
	glVertex2f(0, 0);
	glTexCoord2f(1, 0);
	glVertex2f(s, 0);
	glTexCoord2f(1, 1);
	glVertex2f(s, s);
	glTexCoord2f(0, 1);
	glVertex2f(0, s);
	glEnd();
	glPopMatrix();
	gc.triangleDrawn += 2;
      }
      PopProjection();
    }
  }
  glDisable(GL_TEXTURE_2D);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
}


Module void
drawFlares(Light * light, Float32 transp, MirrorMode m)
{
  Float32 s, alpha, n, hw, hh;
  A3D color;
  A2D flareAxis;
  Flare *fl;
  Int32 i;

  if (!gc.flare || !gc.light || !MirrorImage(m, gc.mirrorMode))
    return;

  if (light->fade[m] >= 1)
    return;

  setRasterOrthoProj();
  hw = gc.winWidth * .5;
  hh = gc.winHeight * .5;
  X(flareAxis) = hw - light->rx[m];
  Y(flareAxis) = hh - light->ry[m];
  n = Min(hh, hw);
  UAssert(n > 0);
  alpha = transp * (1 - light->fade[m]) * (1 - sqrt((X(flareAxis) * X(flareAxis) + Y(flareAxis) * Y(flareAxis))) / n);
  if (alpha > 0) {
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_LIGHTING);
    glEnable(GL_BLEND);
    glEnable(GL_TEXTURE_2D);
    glBlendFunc(GL_ONE, GL_ONE);

    glPushMatrix();
    glLoadIdentity();
    for (i = 0; i < gc.flareNb; i++) {
      fl = &(gc.flares[i]);
      glPushMatrix();
      glLoadIdentity();
      glBindTexture(GL_TEXTURE_2D, fl->texI);
      s = fl->size * light->rs[m];
      R(color) = R(fl->color) * alpha;
      G(color) = G(fl->color) * alpha;
      B(color) = B(fl->color) * alpha;
      glColor3fv(color);
      glTranslatef((gc.winWidth - s) * .5 + X(flareAxis) * fl->pos, (gc.winHeight - s) * .5 + Y(flareAxis) * fl->pos, 0);
      glBegin(GL_QUADS);
      glTexCoord2f(0.0, 0.0);
      glVertex3f(0, 0, 0);
      glTexCoord2f(1.0, 0.0);
      glVertex3f(s, 0, 0);
      glTexCoord2f(1.0, 1.0);
      glVertex3f(s, s, 0);
      glTexCoord2f(0.0, 1.0);
      glVertex3f(0, s, 0);
      glEnd();
      glPopMatrix();
    }
    glPopMatrix();
    glEnable(GL_DEPTH_TEST);
    glDisable(GL_TEXTURE_2D);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  }
  PopProjection();
}


Module void
drawSurfaceNorm(Surface * c)
{
  Float32 *eval, *norm, scale;
  Int32 i, j;
  A3D ne;
  A4D colorTop, colorBot;

  if (!gc.norm)
    return;

  A4DSet(colorTop, 1 - R(c->colorD), 1 - G(c->colorD), 1 - B(c->colorD), 1);
  A4DCp(gc.normColor, colorBot);
  scale = gc.boxSize * 0.033;
  glDisable(GL_LIGHTING);
  glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
  glEnable(GL_LINE_SMOOTH);
  glShadeModel(GL_SMOOTH);
  eval = c->eval;
  norm = c->norm;
  glBegin(GL_LINES);
  for (i = 0; i < c->uRes - 1; i++) {
    for (j = 0; j < c->vRes; j++) {
      glColor4fv(colorBot);
      glVertex3fv(eval);
      glColor4fv(colorTop);
      A3DSet(ne, X(eval) + scale * X(norm), Y(eval) + scale * Y(norm), Z(eval) + scale * Z(norm));
      glVertex3fv(ne);
      eval += 3;
      norm += 3;
    }
  }
  glEnd();
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  glDisable(GL_LINE_SMOOTH);

  if (!gc.smooth)
    glShadeModel(GL_FLAT);
}

Module void
drawSurfaceProjPoint(Surface * c)
{
  Float32 x, *eval;
  Int32 i, j;

  x = gc.boxSize * .98;
  glDisable(GL_LIGHTING);
  glBegin(GL_POINTS);
  glColor4fv(gc.pointColor);
  eval = c->eval;
  for (i = 0; i < c->uRes - 1; i++) {
    for (j = 0; j < c->vRes; j++) {
      glVertex3f(x, *(eval + 1), *(eval + 2));
      eval += 3;
    }
  }
  glEnd();
  glEnable(GL_LIGHTING);
}

Module void
drawSurfaceProjWire(Surface * c)
{
  Float32 x, *eval1, *eval2;
  Int32 i, j;

  x = gc.boxSize * .98;
  glDisable(GL_LIGHTING);
  glEnable(GL_LINE_SMOOTH);
  glColor4fv(gc.wireColor);
  glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
  eval1 = c->eval;
  eval2 = &(c->eval[3 * c->vRes]);
  for (i = 0; i < c->uRes - 1; i++) {
    glBegin(GL_QUAD_STRIP);
    for (j = 0; j < c->vRes; j++) {
      glVertex3f(x, *(eval1 + 1), *(eval1 + 2));
      glVertex3f(x, *(eval2 + 1), *(eval2 + 2));
      eval1 += 3;
      eval2 += 3;
    }
    glEnd();
  }
  glDisable(GL_LINE_SMOOTH);
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
}

Module void
drawSurfaceProjPlain(Surface * c)
{
  Float32 x, *eval1, *eval2;
  Int32 i, j;

  x = gc.boxSize * .98;
  setMaterial(MatSurfaceProj, GL_FRONT_AND_BACK);
  glEnable(GL_COLOR_MATERIAL);
  glColor4fv(c->colorD);
  glNormal3f(-1, 0, 0);
  eval1 = c->eval;
  eval2 = &(c->eval[3 * c->vRes]);
  for (i = 0; i < c->uRes - 1; i++) {
    glBegin(GL_TRIANGLE_STRIP);
    for (j = 0; j < c->vRes; j++) {
      glVertex3f(x, *(eval1 + 1), *(eval1 + 2));
      glVertex3f(x, *(eval2 + 1), *(eval2 + 2));
      eval1 += 3;
      eval2 += 3;
    }
    glEnd();
  }
  glDisable(GL_COLOR_MATERIAL);
  gc.triangleDrawn += c->uRes * c->vRes * 2;
}

Module void
drawSurfaceProjTexture(Surface * c)
{
  Float32 x, *eval1, *eval2, s, t;
  Int32 i, j;

  x = gc.boxSize * .98;
  setMaterial(MatSurfaceFront, GL_FRONT_AND_BACK);
  glEnable(GL_LIGHTING);
  glEnable(GL_COLOR_MATERIAL);
  glColor4fv(c->colorD);

  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, gc.texIds[gc.texI]);

  glNormal3f(-1, 0, 0);
  eval1 = c->eval;
  eval2 = &(c->eval[3 * c->vRes]);
  for (i = 0; i < c->uRes - 1; i++) {
    glBegin(GL_TRIANGLE_STRIP);
    for (j = 0; j < c->vRes; j++) {
      s = *(eval1 + 1) * 5;
      t = *(eval1 + 2) * 5;
      glTexCoord2f(s, t);
      glVertex3f(x, *(eval1 + 1), *(eval1 + 2));
      s = *(eval2 + 1) * 5;
      t = *(eval2 + 2) * 5;
      glTexCoord2f(s, t);
      glVertex3f(x, *(eval2 + 1), *(eval2 + 2));
      eval1 += 3;
      eval2 += 3;
    }
    glEnd();
  }
  glDisable(GL_TEXTURE_2D);
  glDisable(GL_COLOR_MATERIAL);
  gc.triangleDrawn += c->uRes * c->vRes * 2;
}

Module void
drawSurfaceProj(Surface * c)
{

  if (!gc.proj)
    return;

  glPushMatrix();
  glTranslatef(0, 0, -(c->zMin / c->bbDim) + gc.boxSize / 20);
  switch (gc.surfaceMode) {
  case SModePoint:
    drawSurfaceProjPoint(c);
    break;
  case SModeWireV:
  case SModeWireH:
    drawSurfaceProjWire(c);
    break;
  case SModeCMap:
  case SModePlain:
    if (gc.texI > 0)
      drawSurfaceProjTexture(c);
    else
      drawSurfaceProjPlain(c);
    break;
  }
  glPopMatrix();
}

Module void
drawSurfacePoint(Surface * c)
{
  Float32 *eval;
  Int32 i, j;

  glDisable(GL_LIGHTING);
  glBegin(GL_POINTS);
  glColor4fv(gc.pointColor);
  eval = c->eval;
  for (i = 0; i < c->uRes - 1; i++) {
    for (j = 0; j < c->vRes; j++) {
      glVertex3fv(eval);
      eval += 3;
    }
  }
  glEnd();
  glEnable(GL_LIGHTING);
}

Module void
drawSurfaceWireV(Surface * c)
{
  Float32 *eval1, *eval2;
  Int32 i, j;

  glDisable(GL_LIGHTING);
  glColor4fv(gc.wireColor);
  glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
  glEnable(GL_LINE_SMOOTH);
  eval1 = c->eval;
  eval2 = &(c->eval[3 * c->vRes]);
  for (i = 0; i < c->uRes - 1; i++) {
    glBegin(GL_QUAD_STRIP);
    for (j = 0; j < c->vRes; j++) {
      glVertex3fv(eval1);
      glVertex3fv(eval2);
      eval1 += 3;
      eval2 += 3;
    }
    glEnd();
  }
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  glDisable(GL_LINE_SMOOTH);
}

/* wire lines but with hiden lines removal
 * brute force o:)
 * draw a transparent surface then decal and redraw line with Z test against preset surface
 * should be optimized
 */
Module void
drawSurfaceWireH(Surface * c)
{
  Float32 *eval1, *eval2;
  Int32 i, j;

  glDisable(GL_LIGHTING);
  glShadeModel(GL_FLAT);
  glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);
  glColor4fv(gc.backgroundColor);
  glPolygonOffset(1.0, 1.0);
  glEnable(GL_POLYGON_OFFSET_FILL);
  eval1 = c->eval;
  eval2 = &(c->eval[3 * c->vRes]);
  for (i = 0; i < c->uRes - 1; i++) {
    glBegin(GL_QUAD_STRIP);
    for (j = 0; j < c->vRes; j++) {
      glVertex3fv(eval1);
      glVertex3fv(eval2);
      eval1 += 3;
      eval2 += 3;
    }
    glEnd();
  }
  glDisable(GL_POLYGON_OFFSET_FILL);

  glColor4fv(gc.wireColor);
  glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
  glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
  glEnable(GL_LINE_SMOOTH);
  eval1 = c->eval;
  eval2 = &(c->eval[3 * c->vRes]);
  for (i = 0; i < c->uRes - 1; i++) {
    glBegin(GL_QUAD_STRIP);
    for (j = 0; j < c->vRes; j++) {
      glVertex3fv(eval1);
      glVertex3fv(eval2);
      eval1 += 3;
      eval2 += 3;
    }
    glEnd();
  }
  gc.triangleDrawn += c->uRes * c->vRes * 2;
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  glDisable(GL_LINE_SMOOTH);
}

Module void
drawSurfacePlain(Surface * c)
{
  Float32 *eval1, *eval2, *norm1, *norm2;
  Int32 i, j;

  setMaterial(MatSurfaceFront, GL_FRONT_AND_BACK);
  glEnable(GL_COLOR_MATERIAL);
  glColor4fv(c->colorD);

  eval1 = c->eval;
  eval2 = &(c->eval[3 * c->vRes]);
  norm1 = c->norm;
  norm2 = &(c->norm[3 * c->vRes]);
  for (i = 0; i < c->uRes - 1; i++) {
    glBegin(GL_QUAD_STRIP);
    for (j = 0; j < c->vRes; j++) {
      glNormal3fv(norm1);
      glVertex3fv(eval1);
      eval1 += 3;
      norm1 += 3;
      glNormal3fv(norm2);
      glVertex3fv(eval2);
      eval2 += 3;
      norm2 += 3;
    }
    glEnd();
  }
  gc.triangleDrawn += c->uRes * c->vRes * 2;
  glDisable(GL_COLOR_MATERIAL);
}

Module void
drawSurfaceCMap(Surface * c)
{
  Float32 *eval1, *eval2, *norm1, *norm2, zd, zm, a;
  Int32 i, j, ci;
  A4D iColor;

  setMaterial(MatSurfaceFront, GL_FRONT_AND_BACK);
  glEnable(GL_COLOR_MATERIAL);

  zm = c->bbDim / c->zDim;
  zd = -c->zMin / c->zDim;

  eval1 = c->eval;
  eval2 = &(c->eval[3 * c->vRes]);
  norm1 = c->norm;
  norm2 = &(c->norm[3 * c->vRes]);
  for (i = 0; i < c->uRes - 1; i++) {
    glBegin(GL_QUAD_STRIP);
    for (j = 0; j < c->vRes; j++) {
      glNormal3fv(norm1);
      a = (1 - ((*(eval1 + 2)) * zm + zd)) * .999;	/* [0..1[ */
      ci = a * (ColorMapNb - 1);
      A4DInterp(colorMap[ci], colorMap[ci + 1], fmod(a, (ColorMapNb - 1)), iColor);
      glColor4fv(iColor);
      glVertex3fv(eval1);
      eval1 += 3;
      norm1 += 3;
      glNormal3fv(norm2);
      a = (1 - ((*(eval2 + 2)) * zm + zd)) * .999;	/* [0..1[ */
      ci = a * (ColorMapNb - 1);
      A4DInterp(colorMap[ci], colorMap[ci + 1], fmod(a, (ColorMapNb - 1)), iColor);
      glColor4fv(iColor);
      glVertex3fv(eval2);
      eval2 += 3;
      norm2 += 3;
    }
    glEnd();
  }
  glDisable(GL_COLOR_MATERIAL);
  gc.triangleDrawn += c->uRes * c->vRes * 2;
}

Module void
drawSurfaceTexture(Surface * c)
{
  Float32 *eval1, *eval2, *norm1, *norm2, s, t;
  Int32 i, j;

  setMaterial(MatSurfaceFront, GL_FRONT_AND_BACK);
  glEnable(GL_LIGHTING);
  glEnable(GL_COLOR_MATERIAL);
  glColor4fv(c->colorD);

  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, gc.texIds[gc.texI]);

  glEnable(GL_ALPHA_TEST);
  glAlphaFunc(GL_GREATER, 0.5);

  eval1 = c->eval;
  eval2 = &(c->eval[3 * c->vRes]);
  norm1 = c->norm;
  norm2 = &(c->norm[3 * c->vRes]);
  for (i = 0; i < c->uRes - 1; i++) {
    glBegin(GL_QUAD_STRIP);
    for (j = 0; j < c->vRes; j++) {
      glNormal3fv(norm1);
      s = (*eval1) * 4;
      t = (*(eval1 + 1)) * 4;
      glTexCoord2f(s, t);
      glVertex3fv(eval1);
      eval1 += 3;
      norm1 += 3;
      glNormal3fv(norm2);
      s = (*eval2) * 4;
      t = *(eval2 + 1) * 4;
      glTexCoord2f(s, t);
      glVertex3fv(eval2);
      eval2 += 3;
      norm2 += 3;
    }
    glEnd();
  }
  glDisable(GL_ALPHA_TEST);
  gc.triangleDrawn += c->uRes * c->vRes * 2;
  glDisable(GL_TEXTURE_2D);
  glDisable(GL_COLOR_MATERIAL);
}

Module void
drawSurface(Surface * c)
{
  glPushMatrix();
  // (shift a bit to avoid ground intrusion)
  glTranslatef(0, 0, -(c->zMin / c->bbDim) + gc.boxSize / 20);
  glEnable(GL_RESCALE_NORMAL);
  switch (gc.surfaceMode) {
  case SModePoint:
    drawSurfacePoint(c);
    break;
  case SModeWireV:
    drawSurfaceWireV(c);
    break;
  case SModeWireH:
    drawSurfaceWireH(c);
    break;
  case SModePlain:
    /* 0 means no texture */
    if (gc.texI > 0) {
      drawSurfaceTexture(c);
    }
    else {
      drawSurfacePlain(c);
    }
    break;
  case SModeCMap:
    drawSurfaceCMap(c);
    break;
  }
  glDisable(GL_RESCALE_NORMAL);
  drawSurfaceNorm(c);
  glPopMatrix();
}

Module void
drawScene(void)
{
  Surface *c;

  glPushMatrix();
  glLoadIdentity();

  // clear if needed
  if (!gc.wall) {
    glClearColor(A4DGet(gc.backgroundColor));
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  }

  // start stats
  if (!(gc.frame % FrameStats)) {
    DWChronoReset(&gc.chrono, DWChronoReal);
    gc.triangleDrawn = 0;
  }

  // instant frame rate
  DWChronoReset(&gc.frameChronos[gc.fci], DWChronoReal);

  c = gc.surface;

  setObs(&gc.obs);
  setFog();

  if (gc.smooth)
    glShadeModel(GL_SMOOTH);
  else
    glShadeModel(GL_FLAT);

  glEnable(GL_LIGHTING);

  // draw reflections
  if (gc.mirrorMode && (gc.mirrorTransp != 0)) {
    if (gc.mirrorMode & MirrorWall) {
      glPushMatrix();
      glTranslatef(0, 2 * gc.boxSize, 0);

      if (gc.mirrorMode & MirrorGround) {
	glPushMatrix();
	glScalef(1, -1, -1);
	setLights(MirrorGroundWall);
	drawWalls();
	drawGrid(WallRight);
	drawBanner();
	drawSurfaceProj(c);
	pickZLights(MirrorGroundWall);
	drawTri();
	setClipPlane(&gc.clipPlane);
	drawSurface(c);
	unsetClipPlane(&gc.clipPlane);
	checkZLights(MirrorGroundWall);
	drawClipPlane(c, &gc.clipPlane);
	drawLights(MirrorGroundWall);
	drawMirror(MirrorWall);
	glPopMatrix();
      }

      glPushMatrix();
      glScalef(1, -1, 1);
      setLights(MirrorWall);
      drawMirror(MirrorGround);
      drawWalls();
      drawGrid(WallRight);
      drawBanner();
      drawGrid(WallGround);
      drawSurfaceProj(c);
      pickZLights(MirrorWall);
      pickZLights(MirrorGroundWall);
      drawTri();
      setClipPlane(&gc.clipPlane);
      drawSurface(c);
      unsetClipPlane(&gc.clipPlane);
      checkZLights(MirrorWall);
      checkZLights(MirrorGroundWall);
      drawClipPlane(c, &gc.clipPlane);
      drawLights(MirrorWall);
      glPopMatrix();
      glPopMatrix();
    }

    if (gc.mirrorMode & MirrorGround) {
      glPushMatrix();
      glScalef(1, 1, -1);
      setLights(MirrorGround);
      drawWalls();
      drawGrid(WallRight);
      drawBanner();
      drawSurfaceProj(c);
      pickZLights(1);
      pickZLights(MirrorWall);
      pickZLights(MirrorGroundWall);
      drawTri();
      setClipPlane(&gc.clipPlane);
      drawSurface(c);
      unsetClipPlane(&gc.clipPlane);
      checkZLights(MirrorGround);
      checkZLights(MirrorWall);
      checkZLights(MirrorGroundWall);
      drawClipPlane(c, &gc.clipPlane);
      drawLights(MirrorGround);
      glPopMatrix();
    }
  }

  // set light position (use current modelview)
  setLights(MirrorNone);
  // draw plain walls
  drawWalls();
  // draw ground/wall mirrors if needed
  drawMirror(MirrorWall);
  drawMirror(MirrorGround);
  // draw grids on wall and ground
  drawGrid(WallRight);
  drawGrid(WallGround);
  // draw scrolling banner 
  drawBanner();
  // draw surface wall projection
  drawSurfaceProj(c);
  // preset and pick Z of light for occlusion detection
  pickZLights(MirrorNone);
  pickZLights(MirrorGround);
  pickZLights(MirrorWall);
  pickZLights(MirrorGroundWall);
  // draw x/y/z 
  drawTri();
  // cut surface
  setClipPlane(&gc.clipPlane);
  // draw main surface 
  drawSurface(c);
  // remove clip plane
  unsetClipPlane(&gc.clipPlane);
  // check Z of light to detect occlusion
  checkZLights(MirrorNone);
  checkZLights(MirrorGround);
  checkZLights(MirrorWall);
  checkZLights(MirrorGroundWall);
  // draw the cuting plane
  drawClipPlane(c, &gc.clipPlane);
  // draw lights 
  drawLights(MirrorNone);
  // draw all flares (to track lights and their mirror images)
  drawFlares(&gc.lights[0], .1, MirrorGroundWall);
  drawFlares(&gc.lights[0], .3, MirrorWall);
  drawFlares(&gc.lights[0], .3, MirrorGround);
  drawFlares(&gc.lights[0], 1, MirrorNone);

  // on more frame done :)
  gc.frame++;

  // draw help+fps
  drawHelp();

  // end
  glutSwapBuffers();
  glPopMatrix();

  // update all animated objects
  updateFrameRate();
  updateScene();
}

Module void
updateFrameRate(void)
{
  unsigned long ms;
  gc.fci = (gc.fci + 1) % FrameRateCount;
  ms = DWChronoGet(&gc.frameChronos[gc.fci]);
  gc.spf = ms / (1000.0 * FrameRateCount);
}

Module void
updateObs(Observer * obs)
{
  A3D d;

  if (!A3DZeroP(obs->sSpeed)) {
    A3DCp(obs->sSpeed, d);
    A3DMult(d, gc.spf);
    A3DAddA3D(obs->sPos, d);
    if (obs->sPos[0] <= 0) {
      obs->sPos[0] = 0;
      obs->sSpeed[0] = 0;
    }
    if (obs->sPos[2] >= Pi / 2 - 0.01) {
      obs->sPos[2] = Pi / 2 - 0.01;
      obs->sSpeed[2] = 0;
    }
    else if (obs->sPos[2] <= 0.1) {
      obs->sPos[2] = 0.1;
      obs->sSpeed[2] = 0;
    }
    obs->changed = UTrue;
  }
}

Module void
updateClipPlane(ClipPlane * cp)
{
  A3D d;

  if (!A3DZeroP(cp->sSpeed)) {
    A3DCp(cp->sSpeed, d);
    A3DMult(d, gc.spf);
    A3DAddA3D(cp->sPos, d);
    if (Abs(cp->sPos[0]) > .5) {
      cp->sPos[0] = copysign(.5, cp->sPos[0]);
      cp->sSpeed[0] = 0;
    }
  }
}

Module void
updateLight(Light * l)
{
  A3D d;
  MirrorMode m;

  if (!A3DZeroP(l->sSpeed)) {
    A3DCp(l->sSpeed, d);
    A3DMult(d, gc.spf);
    A3DAddA3D(l->sPos, d);
    if (l->sPos[2] <= 0.25) {
      l->sPos[2] = 0.25;
      l->sSpeed[2] *= -1;
    }
    else if (l->sPos[2] >= Pi - 0.25) {
      l->sPos[2] = Pi - 0.25;
      l->sSpeed[2] *= -1;
    }
  }
  for (m = 0; m < MirrorLast; m++) {
    if (l->fadeSpeed[m] != 0) {
      l->fade[m] += gc.spf * l->fadeSpeed[m];
      if (l->fade[m] > 1) {
	l->fade[m] = 1;
	l->fadeSpeed[m] = 0;
      }
      else if (l->fade[m] < 0) {
	l->fade[m] = 0;
	l->fadeSpeed[m] = 0;
      }
    }
  }
}

Module void
updateMirror(void)
{
  if (gc.mirrorMode && (gc.mirrorTranspInc != 0.0)) {
    gc.mirrorTransp += gc.mirrorTranspInc * gc.spf;
    if (gc.mirrorTransp >= 1) {
      gc.mirrorTransp = 1;
      gc.mirrorTranspInc = 0;
    }
    else if (gc.mirrorTransp <= 0) {
      gc.mirrorTransp = 0;
      gc.mirrorTranspInc = 0;
    }
    A4DInterp(gc.materials[MatPerfectMirror].ambient, gc.materials[MatWall].ambient, gc.mirrorTransp, gc.materials[MatMirror].ambient);
    A4DInterp(gc.materials[MatPerfectMirror].diffuse, gc.materials[MatWall].diffuse, gc.mirrorTransp, gc.materials[MatMirror].diffuse);
  }
}

Module Int32
updateParam(Parameter * p)
{
  if (p->speed) {
    *(p->param) += p->speed * gc.spf;
    if (p->maxVal != p->minVal) {
      if (*(p->param) > p->maxVal) {
	*(p->param) = p->maxVal;
	p->speed = 0;
      }
      if (*(p->param) < p->minVal) {
	*(p->param) = p->minVal;
	p->speed = 0;
      }
    }
    return 1;
  }
  return 0;
}

Module void
updateSurface(Surface * c)
{
  Int32 p = 0, i;

  for (i = 0; i < c->paramNb; i++) {
    p += updateParam(&(c->param[i]));
  }
  if (p > 0)
    fillSurface(c);
}

Module void
updateBanner(void)
{
  if (gc.banner) {
    gc.bannerOffset += gc.bannerSpeed * gc.spf;
  }
}

Module void
updateScene(void)
{
  updateObs(&gc.obs);
  updateSurface(gc.surface);
  updateLight(&(gc.lights[0]));
  updateMirror();
  updateClipPlane(&(gc.clipPlane));
  updateBanner();
}

Module void
reshapeWindow(int width, int height)
{
  gc.winWidth = width;
  gc.winHeight = height;
  glViewport(gc.winOrigX, gc.winOrigY, gc.winWidth, gc.winHeight);
  glGetIntegerv(GL_VIEWPORT, gc.viewport);	/* for later gluprjetc */
}

Module void
speedObs(Observer * obs, Float32 rs, Float32 ts, Float32 ps)
{
  obs->sSpeed[0] += rs * ObsRhoSpeedInc;
  obs->sSpeed[1] += ts * ObsThetaSpeedInc;
  obs->sSpeed[2] += ps * ObsPhiSpeedInc;
}

Module void
speedParameter(Parameter * p, Float32 pis)
{
  p->speed += p->speedInc * pis;
}

Module void
speedLight(Light * l, Float32 ts, Float32 ps)
{
  l->sSpeed[1] += ts * LightThetaSpeedInc;
  l->sSpeed[2] += ps * LightPhiSpeedInc;
}

Module void
speedPlane(ClipPlane * cp, Float32 rs, Float32 ts)
{
  cp->sSpeed[0] += rs * PlaneRhoSpeedInc;
  cp->sSpeed[1] += ts * PlaneThetaSpeedInc;
  cp->sSpeed[2] = 0;
}

Module void
specialEventCB(int key, int x, int y)
{
  int mod;

  switch (key) {
  case GLUT_KEY_F1:
    gc.help = (!gc.help);
    break;
  }

  mod = glutGetModifiers();
  if (!mod) {
    switch (key) {
    case GLUT_KEY_LEFT:
      speedObs(&gc.obs, 0, -1, 0);
      break;
    case GLUT_KEY_RIGHT:
      speedObs(&gc.obs, 0, 1, 0);
      break;
    case GLUT_KEY_DOWN:
      speedObs(&gc.obs, 0, 0, -1);
      break;
    case GLUT_KEY_UP:
      speedObs(&gc.obs, 0, 0, 1);
      break;
    }
  }
  else if (mod == GLUT_ACTIVE_SHIFT) {
    Surface *c;
    c = gc.surface;
    switch (key) {
    case GLUT_KEY_LEFT:
      speedParameter(&(c->param[c->curPa]), -1);
      break;
    case GLUT_KEY_RIGHT:
      speedParameter(&(c->param[c->curPa]), 1);
      break;
    case GLUT_KEY_DOWN:
      speedParameter(&(c->param[c->curPb]), -1);
      break;
    case GLUT_KEY_UP:
      speedParameter(&(c->param[c->curPb]), 1);
      break;
    }
  }
  else if (mod == GLUT_ACTIVE_ALT) {
    Light *l;
    l = &(gc.lights[0]);
    switch (key) {
    case GLUT_KEY_LEFT:
      speedLight(l, 1, 0);
      break;
    case GLUT_KEY_RIGHT:
      speedLight(l, -1, 0);
      break;
    case GLUT_KEY_DOWN:
      speedLight(l, 0, -1);
      break;
    case GLUT_KEY_UP:
      speedLight(l, 0, 1);
      break;
    }
  }
  else if (mod == GLUT_ACTIVE_CTRL) {
    ClipPlane *cp;
    cp = &(gc.clipPlane);
    switch (key) {
    case GLUT_KEY_LEFT:
      speedPlane(cp, 0, -1);
      break;
    case GLUT_KEY_RIGHT:
      speedPlane(cp, 0, 1);
      break;
    case GLUT_KEY_DOWN:
      speedPlane(cp, -1, 0);
      break;
    case GLUT_KEY_UP:
      speedPlane(cp, 1, 0);
      break;
    }
  }
}

Module void
setPredefinedMode(int i)
{
  Int32 or, j = 0;
  Surface *c;

  gc.surfaceMode = (SurfaceMode) predefinedModes[i][j++];
  gc.smooth = (UBool) predefinedModes[i][j++];
  gc.wall = (UBool) predefinedModes[i][j++];
  switchLight(predefinedModes[i][j++]);
  gc.wallLight = (UBool) predefinedModes[i][j++];
  gc.clip = (UBool) predefinedModes[i][j++];
  gc.mirrorMode = (Int32) predefinedModes[i][j++];
  if (gc.mirrorMode) {
    gc.mirrorTranspInc += MirrorTranspInc;
  }
  gc.proj = (UBool) predefinedModes[i][j++];
  gc.norm = (UBool) predefinedModes[i][j++];
  gc.texI = (Int32) predefinedModes[i][j++];
  gc.fog = (UBool) predefinedModes[i][j++];
  gc.fogChanged = UTrue;
  gc.tri = (UBool) predefinedModes[i][j++];
  gc.grid = (UBool) predefinedModes[i][j++];
  gc.banner = (UBool) predefinedModes[i][j++];
  gc.flare = (UBool) predefinedModes[i][j++];
  gc.wallTexture = (UBool) predefinedModes[i][j++];
  c = gc.surface;
  if (c) {
    or = c->uRes;
    c->vRes = c->uRes = (Int32) predefinedModes[i][j++];
    if (or != c->uRes) {
      reinitSurface(c);
    }
  }
}

Module void
screenDump(void)
{
  static Int32 saved = 0;
  char name[64];
  UByte *image;

  image = UMalloc(gc.winWidth * gc.winHeight * 3);
  glReadPixels(0, 0, gc.winWidth, gc.winHeight, GL_RGB, GL_UNSIGNED_BYTE, image);
  sprintf(name, "freedump-%d.png", saved++);
  imageSave(name, image, gc.winWidth, gc.winHeight);
}

Module void
switchLight(UBool nl)
{
  Int32 m;

  if (gc.light == nl)
    return;

  gc.light = nl;
  for (m = 0; m < MirrorLast; m++) {
    if (nl) {
      /* light was off... turn on smoothly */
      gc.lights[0].fadeSpeed[m] = -LightFadeSpeedInc;
      gc.lights[0].fade[m] = 1;
    }
    else {
      /* light was on... turn off smoothly */
      gc.lights[0].fadeSpeed[m] = LightFadeSpeedInc;
      gc.lights[0].fade[m] = 0;
    }
  }
}

Module void
keyEventCB(unsigned char key, int x, int y)
{
  Int32 pi, mod;
  Surface *c;

  c = gc.surface;
  mod = glutGetModifiers();
  if ((key >= '0') && (key <= '9')) {
    pi = key - '0' - 1;
    if ((pi < c->paramNb) && (pi != c->curPa)) {
      c->curPb = c->curPa;
      c->curPa = pi;
    }
  }
  switch (key) {
  case 27:			/* Escape */
    exit(0);
    break;
  case '?':
    gc.help = (!gc.help);
    break;
  case 'z':
    reinitPos();
    break;
  case '+':
    if (!(mod & GLUT_ACTIVE_ALT)) {
      gc.currentMode = (gc.currentMode + 1) % ModeNb;
      setPredefinedMode(gc.currentMode);
    }
    else {
      gc.surfaceI = (gc.surfaceI + 1) % gc.surfaceNb;
      gc.surface = &(gc.surfaces[gc.surfaceI]);
      reinitSurface(gc.surface);
    }
    break;
  case '-':
    if (!(mod & GLUT_ACTIVE_ALT)) {
      gc.currentMode--;
      if (gc.currentMode < 0)
	gc.currentMode = ModeNb - 1;
      setPredefinedMode(gc.currentMode);
    }
    else {
      gc.surfaceI--;
      if (gc.surfaceI < 0)
	gc.surfaceI = gc.surfaceNb - 1;
      gc.surface = &(gc.surfaces[gc.surfaceI]);
      reinitSurface(gc.surface);
    }
    break;
  case 's':
    gc.smooth = (!gc.smooth);
    break;
  case 'w':
    gc.wall = (!gc.wall);
    break;
  case 'W':
    gc.wallTexture = (!gc.wallTexture);
    break;
  case 'v':
    switchLight(!gc.light);
    break;
  case 'l':
    gc.wallLight = (!gc.wallLight);
    break;
  case 'p':
    gc.proj = (!gc.proj);
    break;
  case 'c':
    gc.clip = (!gc.clip);
    break;
  case 'm':
    gc.mirrorMode++;
    if (gc.mirrorMode >= MirrorLast)
      gc.mirrorMode = 0;
    if (gc.mirrorMode && (gc.mirrorTranspInc == 0) && (gc.mirrorTransp == 0)) {
      gc.mirrorTranspInc += MirrorTranspInc;
    }
    break;
  case 'M':
    gc.mirrorMode--;
    if (gc.mirrorMode < 0)
      gc.mirrorMode = MirrorLast - 1;
    if (gc.mirrorMode && (gc.mirrorTranspInc == 0) && (gc.mirrorTransp == 0)) {
      gc.mirrorTranspInc += MirrorTranspInc;
    }
    break;
  case 'f':
    gc.fog = (!gc.fog);
    gc.fogChanged = UTrue;
    break;
  case 'b':
    gc.banner = (!gc.banner);
    break;
  case 'F':
    gc.flare = (!gc.flare);
    break;
  case 'a':
    gc.tri = (!gc.tri);
    break;
  case 'n':
    gc.norm = (!gc.norm);
    break;
  case 'g':
    gc.grid = (!gc.grid);
    break;
  case 'i':
    gc.mirrorTranspInc += MirrorTranspInc;
    if (gc.mirrorMode == 0) {
      gc.mirrorMode = MirrorGround;
    }
    break;
  case 'I':
    gc.mirrorTranspInc -= MirrorTranspInc;
    break;
  case 't':
    gc.texI++;
    if (gc.texI >= TexLast)
      gc.texI = 0;
    break;
  case 'T':
    gc.texI--;
    if (gc.texI < 0)
      gc.texI = TexLast - 1;
    break;
  case 'd':
    speedObs(&gc.obs, 1, 0, 0);
    break;
  case 'D':
    speedObs(&gc.obs, -1, 0, 0);
    break;
  case 'r':
    gc.surface->uResReal = gc.surface->uResReal * SurfaceResMult;
    gc.surface->vResReal = gc.surface->vResReal * SurfaceResMult;
    gc.surface->uRes = (Int32) gc.surface->uResReal;
    gc.surface->vRes = (Int32) gc.surface->vResReal;
    reinitSurface(gc.surface);
    break;
  case 'R':
    gc.surface->uResReal = gc.surface->uResReal / SurfaceResMult;
    gc.surface->vResReal = gc.surface->vResReal / SurfaceResMult;
    gc.surface->uResReal = Max(1, gc.surface->uResReal);
    gc.surface->vResReal = Max(1, gc.surface->vResReal);
    gc.surface->uRes = (Int32) gc.surface->uResReal;
    gc.surface->vRes = (Int32) gc.surface->vResReal;
    reinitSurface(gc.surface);
    break;
  case 'S':
    screenDump();
    break;
  case ' ':{
      Surface *c;
      Parameter *p;
      Int32 i;

      A3DZero(gc.obs.sSpeed);
      A3DZero(gc.clipPlane.sSpeed);
      c = gc.surface;
      for (i = 0; i < c->paramNb; i++) {
	p = &(c->param[i]);
	p->speed = 0;
      }
      A3DZero(gc.lights[0].sSpeed);
      break;
    }
#ifdef XMESA
  case '!':
    gc.fullScreen = (!gc.fullScreen);
    XMesaSetFXmode(gc.fullScreen ? XMESA_FX_FULLSCREEN : XMESA_FX_WINDOW);
    break;
#endif
  }

}

Module void
fillSurface(Surface * c)
{
  Float32 u, v, x, y, z, ui, vi, *eval, xMin, xMax, yMin, yMax, zMin, zMax,
    bbDim, s, x2, y2, z2, *norm, ueps, veps;
  Int32 i, j;
  A3D t1, t2;

  xMin = yMin = zMin = FLT_MAX;
  xMax = yMax = zMax = -FLT_MAX;
  ui = (c->uMax - c->uMin) / (c->uRes - 1);
  vi = (c->vMax - c->vMin) / (c->vRes - 1);
  ueps = ui * NormEpsilon;
  veps = vi * NormEpsilon;
  eval = c->eval;
  norm = c->norm;
  u = c->uMin;
  for (i = 0; i < c->uRes; i++) {
    v = c->vMin;
    for (j = 0; j < c->vRes; j++) {
      c->f(c, u, v, &x, &y, &z);
      *eval++ = x;
      *eval++ = y;
      *eval++ = z;
      if (x < xMin)
	xMin = x;
      if (x > xMax)
	xMax = x;
      if (y < yMin)
	yMin = y;
      if (y > yMax)
	yMax = y;
      if (z < zMin)
	zMin = z;
      if (z > zMax)
	zMax = z;
      c->f(c, u + ueps, v, &x2, &y2, &z2);
      A3DSet(t1, x2 - x, y2 - y, z2 - z);
      c->f(c, u, v + veps, &x2, &y2, &z2);
      A3DSet(t2, x2 - x, y2 - y, z2 - z);
      A3DNormCrossProd(t1, t2, norm);
      //      A3DPrint(n);
      norm += 3;
      v += vi;
    }
    u += ui;
  }
  c->xMin = xMin;
  c->xMax = xMax;
  c->yMin = yMin;
  c->yMax = yMax;
  c->zMin = zMin;
  c->zMax = zMax;
  c->xDim = (c->xMax - c->xMin);
  c->yDim = (c->yMax - c->yMin);
  c->zDim = (c->zMax - c->zMin);

  // normalize eval
  bbDim = Max(c->xDim, c->yDim);
  /* fake z is bigger to fit surface in screen ratio */
  bbDim = Max(c->zDim * 1.8, bbDim);
  c->bbDim = bbDim;
  c->scale = s = 1 / (Max(c->bbDim, 0.0001));	/* trouble with empty bb */
  eval = c->eval;
  for (i = 0; i < c->uRes * c->vRes; i++) {
    *eval = *eval * s;
    eval++;
    *eval = *eval * s;
    eval++;
    *eval = *eval * s;
    eval++;
  }
}

Module void
reinitSurface(Surface * c)
{
  Int32 size;

  size = c->uRes * c->vRes;
  UFreeZ(c->eval);
  UFreeZ(c->norm);
  c->eval = UMalloc(size * 3 * sizeof(Float32));
  c->norm = UMalloc(size * 3 * sizeof(Float32));
  fillSurface(c);
}

Module void
initParam(Surface * c, Float32 * param, Float32 speedInc, Float32 defVal, Float32 minVal, Float32 maxVal)
{
  Parameter *p;

  p = &(c->param[c->paramNb++]);
  p->param = param;
  *param = defVal;
  p->speedInc = speedInc;
  p->speed = 0;
  p->defVal = defVal;
  p->minVal = minVal;
  p->maxVal = maxVal;
}

Module void
initSurface(Surface * c)
{
  c->curPa = 0;
  c->curPb = 1;
}

Module void
initSurfaces(void)
{
  Surface *c;
  Int32 i;

  for (i = 0; i < MaxSurfaceNb; i++) {
    c = &(gc.surfaces[i]);
    bzero(c, sizeof(Surface));
    c->uResReal = SurfaceDefaultResolution;
    c->uRes = c->uResReal;
    c->vResReal = SurfaceDefaultResolution;
    c->vRes = c->vResReal;
  }
  i = 0;


  c = &(gc.surfaces[i++]);
  c->f = funcBessel;
  c->name = "Bessel";
  c->uMin = c->vMin = -3 * Pi;
  c->uMax = c->vMax = 3 * Pi;
  A4DSet(c->colorD, .6, .5, .1, 1);
  initParam(c, &(c->p1), ParameterSpeedInc, 0, 0, 0);
  initParam(c, &(c->p2), ParameterSpeedInc, 0, 0, 0);
  initParam(c, &(c->p3), ParameterSpeedInc, .25, 0, 0);
  initParam(c, &(c->p4), ParameterSpeedInc, .1, 0, 0);

  c = &(gc.surfaces[i++]);
  c->f = funcQuad;
  c->name = "Quad";
  c->uMin = c->vMin = -1;
  c->uMax = c->vMax = 1;
  A4DSet(c->colorD, .6, .4, .5, 1);
  initParam(c, &(c->p1), ParameterSpeedInc, 0, 0, 0);
  initParam(c, &(c->p2), ParameterSpeedInc, 0, 0, 0);
  initParam(c, &(c->p3), ParameterSpeedInc, 0, 1, 4);
  initParam(c, &(c->p4), ParameterSpeedInc, 0, 1, 4);

  c = &(gc.surfaces[i++]);
  c->f = funcWaves;
  c->name = "Waves";
  c->uMin = c->vMin = -2 * Pi;
  c->uMax = c->vMax = 2 * Pi;
  A4DSet(c->colorD, .5, .6, .1, 1);
  initParam(c, &(c->p1), ParameterSpeedInc, 0, 0, 0);
  initParam(c, &(c->p2), ParameterSpeedInc, 0, 0, 0);
  initParam(c, &(c->p3), ParameterSpeedInc, 1, 0, 0);
  initParam(c, &(c->p4), ParameterSpeedInc, 1, 0, 0);

  c = &(gc.surfaces[i++]);
  c->f = funcTorus;
  c->name = "Torus";
  c->uMin = c->vMin = -Pi;
  c->uMax = c->vMax = Pi;
  A4DSet(c->colorD, .5, .1, .6, 1);
  initParam(c, &(c->p1), ParameterSpeedInc, .2, 0, 0);
  initParam(c, &(c->p2), ParameterSpeedInc, 1, 0, 0);
  initParam(c, &(c->p3), ParameterSpeedInc, 1, 0, 0);
  initParam(c, &(c->p4), ParameterSpeedInc, 1, 0, 0);
  initParam(c, &(c->p5), ParameterSpeedInc, 1, 0.01, 1);
  initParam(c, &(c->p6), ParameterSpeedInc, 1, 0.01, 1);

  c = &(gc.surfaces[i++]);
  c->f = funcInspiration;
  c->name = "Inspiration";
  c->uResReal *= 3;
  c->uRes *= 3;
  c->uMin = c->vMin = -Pi;
  c->uMax = c->vMax = Pi;
  A4DSet(c->colorD, .6, .1, .5, 1);
  initParam(c, &(c->p1), ParameterSpeedInc, .2, 0, 0);
  initParam(c, &(c->p2), ParameterSpeedInc, 1, 0, 0);
  initParam(c, &(c->p3), ParameterSpeedInc, .5, 0, 0);
  initParam(c, &(c->p4), ParameterSpeedInc, 1, 0, 0);

  c = &(gc.surfaces[i++]);
  c->f = funcBoy;
  c->name = "Boy";
  c->uMin = 0;
  c->vMin = -Pi;
  c->uMax = c->vMax = Pi;
  A4DSet(c->colorD, .1, .5, .6, 1);
  initParam(c, &(c->p1), ParameterSpeedInc / 10, 1, 0, 0);
  initParam(c, &(c->p2), ParameterSpeedInc / 10, 1, 0.01, 1);
  initParam(c, &(c->p3), ParameterSpeedInc / 10, 1, 0.01, 1);

  c = &(gc.surfaces[i++]);
  c->f = funcMonkeySaddle;
  c->name = "Monkey Saddle";
  c->uMin = c->vMin = 0;
  c->uMax = 1;
  c->vMax = 2 * Pi;
  A4DSet(c->colorD, .2, .3, .8, 1);
  initParam(c, &(c->p1), ParameterSpeedInc / 10, 3, 1, 10);
  initParam(c, &(c->p2), ParameterSpeedInc / 10, 1, 0.01, 10);
  initParam(c, &(c->p3), ParameterSpeedInc / 10, 1, 0.01, 1);

  c = &(gc.surfaces[i++]);
  c->f = funcShell;
  c->name = "Shell";
  c->uMin = c->vMin = 0;
  c->uMax = Pi;
  c->vMax = 4 * Pi;
  A4DSet(c->colorD, .8, .2, .3, 1);
  initParam(c, &(c->p1), ParameterSpeedInc / 10, 1.3, 0, 0);
  initParam(c, &(c->p2), ParameterSpeedInc / 10, 1, 0, 0);
  initParam(c, &(c->p3), ParameterSpeedInc / 10, 1, 0, 0);
  initParam(c, &(c->p4), ParameterSpeedInc / 10, 1, 0.01, 1);
  initParam(c, &(c->p5), ParameterSpeedInc / 10, 1, 0.01, 1);

  c = &(gc.surfaces[i++]);
  c->f = funcBohemianDome;
  c->name = "Bohemian Dome";
  c->uMin = c->vMin = 0;
  c->uMax = c->vMax = 2 * Pi;
  A4DSet(c->colorD, .1, .8, .3, 1);
  initParam(c, &(c->p1), ParameterSpeedInc / 10, .5, 0, 0);
  initParam(c, &(c->p2), ParameterSpeedInc / 10, 1.5, 0, 0);
  initParam(c, &(c->p3), ParameterSpeedInc / 10, 1, 0, 0);
  initParam(c, &(c->p4), ParameterSpeedInc / 10, 1, 0.01, 1);
  initParam(c, &(c->p5), ParameterSpeedInc / 10, 1, 0.01, 1);

  c = &(gc.surfaces[i++]);
  c->f = funcAstrodalEllipsoid;
  c->name = "Astrodal Ellipsoid";
  c->uMin = 0;
  c->vMin = -Pi;
  c->uMax = c->vMax = Pi;
  A4DSet(c->colorD, .3, .1, .8, 1);
  initParam(c, &(c->p1), ParameterSpeedInc / 10, 1, 0, 0);
  initParam(c, &(c->p2), ParameterSpeedInc / 10, 1, 0, 0);
  initParam(c, &(c->p3), ParameterSpeedInc / 10, 1, 0, 0);
  initParam(c, &(c->p4), ParameterSpeedInc / 10, 1, 0.01, 1);
  initParam(c, &(c->p5), ParameterSpeedInc / 10, 1, 0.01, 1);

  c = &(gc.surfaces[i++]);
  c->f = funcEnneper;
  c->name = "Enneper";
  c->uMin = c->vMin = -2;
  c->uMax = c->vMax = 2;
  A4DSet(c->colorD, .3, .8, .1, 1);
  initParam(c, &(c->p1), ParameterSpeedInc / 10, 1, 0, 0);
  initParam(c, &(c->p2), ParameterSpeedInc / 10, 1, 0, 0);

  c = &(gc.surfaces[i++]);
  c->f = funcKleinBottle;
  c->name = "Klein's Bottle";
  c->uMin = -2 * Pi;
  c->vMin = -Pi;
  c->uMax = 2 * Pi;
  c->vMax = Pi;
  c->uResReal *= 3;
  c->uRes *= 3;
  A4DSet(c->colorD, .5, .5, .2, 1);
  initParam(c, &(c->p1), ParameterSpeedInc / 10, 1, 0.01, 1);
  initParam(c, &(c->p2), ParameterSpeedInc / 10, 1, 0.01, 1);

  c = &(gc.surfaces[i++]);
  c->f = funcKuen;
  c->name = "Kuen";
  c->uMin = -4;
  c->vMin = .05;
  c->uMax = 4;
  c->vMax = Pi - .05;
  A4DSet(c->colorD, .5, .2, .5, 1);
  initParam(c, &(c->p1), ParameterSpeedInc / 10, 1, 0.01, 1);
  initParam(c, &(c->p2), ParameterSpeedInc / 10, 1, 0.01, 1);


  gc.surfaceNb = i;
  UAssert(gc.surfaceNb < MaxSurfaceNb);
  for (i = 0; i < gc.surfaceNb; i++) {
    initSurface(&(gc.surfaces[i]));
  }
  gc.surfaceI = 0;
  gc.surface = &gc.surfaces[gc.surfaceI];
  reinitSurface(gc.surface);
}

Module Observer *
initObserver(Observer * obs)
{
  bzero(obs, sizeof(Observer));
  A3DSet(obs->sPos, gc.boxSize * .9, -Pi_2 - 0.2, Pi / 6);
  obs->fov = ObsDefFov;
  obs->changed = UTrue;
  return obs;
}

Module void
initLights(void)
{
  Light *l;
  Int32 m;

  A1DSet(gc.lModelLocal, 1.);
  A1DSet(gc.lModelTwoSide, .0);
  A4DSet(gc.lModelAmbient, .1, .1, .1, 1.);

  l = &(gc.lights[0]);
  l->on = UTrue;
  A3DSet(l->sPos, LampDistance * gc.boxSize, 3 * (Pi_4), Pi_4 - 0.1);
  A4DSet(l->ambient, .2, .2, .2, 1.0);
  A4DSet(l->diffuse, 1., 1., 1., 1.0);
  A4DSet(l->specular, 1., 1., 1., 1.0);
  for (m = 0; m < MirrorLast; m++) {
    l->fade[m] = 1;
    l->fadeSpeed[m] = -LightFadeSpeedInc;
    l->hidden[m] = UFalse;
  }
}

Module void
initMaterials(void)
{
  Material *m;
  int i;

  // default plain red material
  m = &(gc.materials[MatDefault]);
  A4DSet(m->ambient, 1., .0, .0, 1.);
  A4DSet(m->diffuse, 1., .0, .0, 1.);
  A4DSet(m->specular, 1., 1., 1., 1.);
  A4DSet(m->emission, .0, .0, .0, 1.);
  A1DSet(m->shininess, 10.);

  for (i = 0; i < MatLast; i++) {
    bcopy(&(gc.materials[MatDefault]), &(gc.materials[i]), sizeof(Material));
  }

  // bright front face
  m = &(gc.materials[MatSurfaceFront]);
  A4DSet(m->ambient, .5, .1, .5, 1.);
  A4DCp(m->ambient, m->diffuse);
  A4DSet(m->specular, 0, 0, 0, 1.);
  A1DSet(m->shininess, 10.);

  // dim backface 
  m = &(gc.materials[MatSurfaceBack]);
  A4DSet(m->ambient, .1, .0, .1, 1.);
  A4DCp(m->ambient, m->diffuse);
  A4DSet(m->specular, .0, .0, .0, 1.);
  A1DSet(m->shininess, 1.);

  // lamp (emissive)
  m = &(gc.materials[MatLight]);
  A4DSet(m->emission, 1., 1., .8, .0);
  A4DSet(m->ambient, .0, .0, .0, 1.);
  A4DSet(m->diffuse, 1., 1., 1., 1.);
  A4DSet(m->specular, .0, .0, .0, 1.);

  // perfect mirror (in fact alpha transluscent)
  m = &(gc.materials[MatPerfectMirror]);
  A4DSet(m->ambient, .1, .1, .2, .45);
  A4DCp(m->ambient, m->diffuse);

  // plain wall 
  m = &(gc.materials[MatWall]);
  A4DSet(m->ambient, .35, .3, .35, 1);
  A4DCp(m->ambient, m->diffuse);
  A4DSet(m->specular, .3, .3, .3, 1);
  A1DSet(m->shininess, 1.);

  // variable mirror starts as a plain wall
  bcopy(&(gc.materials[MatWall]), &(gc.materials[MatMirror]), sizeof(Material));

  // surface projection on wall
  m = &(gc.materials[MatSurfaceProj]);
  A4DSet(m->ambient, .5, .1, .5, 1.);
  A4DCp(m->ambient, m->diffuse);
  A4DSet(m->specular, 1., 1., 1., 1);
  A1DSet(m->shininess, 70.);

  // various lines
  m = &(gc.materials[MatLine]);
  A4DSet(m->ambient, .8, .8, .9, .99);
  A4DSet(m->ambient, .0, .0, .0, 1.);
  A4DSet(m->specular, .0, .0, .0, 1.);

  // gold
  m = &(gc.materials[MatGold]);
  A4DSet(m->ambient, .4, .2, 0, 1.);
  A4DSet(m->diffuse, .9, .5, 0, 1.);
  A4DSet(m->specular, .7, .7, 0., 1.);
  A4DSet(m->emission, .0, .0, .0, 1.);
  A1DSet(m->shininess, 10.);

  // clip plane
  m = &(gc.materials[MatClipPlane]);
  A4DSet(m->ambient, .2, .2, .8, .2);
  A4DCp(m->ambient, m->diffuse);
  A1DSet(m->shininess, 40.);

  // scrolling banner
  m = &(gc.materials[MatBanner]);
  A4DSet(m->ambient, .2, .2, .8, 1);
  A4DCp(m->ambient, m->diffuse);
  A4DSet(m->specular, .01, .01, .01, 1.);
  A1DSet(m->shininess, 1.);
}

Module void
reinitPos(void)
{
  Surface *c;
  Parameter *p;
  Int32 i;

  initLights();
  initObserver(&gc.obs);
  c = gc.surface;
  initSurface(c);
  for (i = 0; i < c->paramNb; i++) {
    p = &(c->param[i]);
    *(p->param) = p->defVal;
  }
  reinitSurface(c);
}

Module void
loadFont(TexFont ** ptxf, char *fileName)
{
  TexFont *txf;

  *ptxf = NULL;
  txf = txfLoadFont(fileName);
  if (txf == NULL) {
    fprintf(stderr, "Problem loading %s, %s\n", fileName, txfErrorString());
    exit(1);
  }
  *ptxf = txf;
}

Module void
initFonts(void)
{
  Int32 i, w, dummy;
  Float32 offset = 0;
  char tmp[2];

  loadFont(&gc.fonts[FontHelp], FontHelpFile);
  loadFont(&gc.fonts[FontBanner], FontBannerFile);
  gc.bannerTextLen = strlen(gc.bannerText);
  gc.bannerOffsets = UMalloc(sizeof(Float32) * (gc.bannerTextLen + 1));
  i = 0;
  tmp[1] = '\0';
  while (gc.bannerText[i]) {
    tmp[0] = gc.bannerText[i];
    txfGetStringMetrics(gc.fonts[FontBanner], tmp, 1, &w, &dummy, &dummy);
    gc.bannerOffsets[i] = offset;
    offset += w;
    i++;
  }
  gc.bannerOffsets[i] = offset;
}

Module void
loadImageTexture(TexId tid, char *fileName, Int32 pixelType)
{
  UByte *bytes;
  GLenum gluerr;
  png_infop img;

  glGenTextures(1, &gc.texIds[tid]);
  glBindTexture(GL_TEXTURE_2D, gc.texIds[tid]);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  if (!(img = imageLoad(fileName, &bytes))) {
    fprintf(stderr, "Error reading a texture (%s).\n", fileName);
    exit(-1);
  }
  if ((gluerr = gluBuild2DMipmaps(GL_TEXTURE_2D, 3, img->width, img->height, pixelType,
				  GL_UNSIGNED_BYTE, (GLvoid *) (bytes)))) {
    fprintf(stderr, "GLULib%s\n", gluErrorString(gluerr));
    exit(-1);
  }
}

Module void
initFlares(void)
{
  char name[128];
  GLenum gluerr;
  UByte *bytes;
  png_infop img;
  Int32 i, f;

  for (i = 0; i < TexFlareNb; i++) {
    glGenTextures(1, &gc.texFlares[i]);
    glBindTexture(GL_TEXTURE_2D, gc.texFlares[i]);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    sprintf(name, "textures/flares/flare%d.png", i);
    if (!(img = imageLoad(name, &bytes))) {
      fprintf(stderr, "Error reading texture %s.\n", name);
      exit(-1);
    }
    if ((gluerr = gluBuild2DMipmaps(GL_TEXTURE_2D, 3, img->width, img->height, GL_LUMINANCE,
				    GL_UNSIGNED_BYTE, (GLvoid *) (bytes)))) {
      fprintf(stderr, "GLULib%s\n", gluErrorString(gluerr));
      exit(-1);
    }
  }

#define SetFlare(_Ti,_Pos,_Sz,_Col,_In) \
  gc.flares[gc.flareNb].texI=gc.texFlares[_Ti];\
  gc.flares[gc.flareNb].pos=_Pos;\
  gc.flares[gc.flareNb].size=_Sz;\
  A4DCp(_Col,gc.flares[gc.flareNb].color);\
  A4DMult(gc.flares[gc.flareNb].color,_In);\
  gc.flareNb++;

  SetFlare(2, .6, 20, ColWhite, 0.6);
  SetFlare(1, 0.5, 50, ColWhite, 0.3);
  SetFlare(3, .25, 60, ColWhite, 0.4);
  SetFlare(3, 0.2, 50, ColWhite, 0.3);
  SetFlare(5, -0.25, 70, ColWhite, 0.5);
  SetFlare(5, -0.4, 20, ColWhite, 0.6);
  SetFlare(5, -0.6, 40, ColWhite, 0.4);
  SetFlare(3, -.85, 20, ColWhite, 0.6);
  SetFlare(0, -1.0, 10, ColWhite, 1.);
#undef SetFlare


  /* copy some flares as shines */
  f = 0;
  gc.texShines[f++] = gc.texFlares[2];
  gc.texShines[f++] = gc.texFlares[4];
  for (i = 0; i < TexShineNb; i++) {
    glGenTextures(1, &gc.texShines[f + i]);
    glBindTexture(GL_TEXTURE_2D, gc.texShines[f + i]);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    sprintf(name, "textures/flares/shine%d.png", i);
    if (!(img = imageLoad(name, &bytes))) {
      fprintf(stderr, "Error reading texture %s.\n", name);
      exit(-1);
    }
    if ((gluerr = gluBuild2DMipmaps(GL_TEXTURE_2D, 3, img->width, img->height, GL_LUMINANCE,
				    GL_UNSIGNED_BYTE, (GLvoid *) (bytes)))) {
      fprintf(stderr, "GLULib%s\n", gluErrorString(gluerr));
      exit(-1);
    }
  }


#define SetShine(_Ti,_Sz,_Cst,_Col,_In) \
  gc.shines[gc.shineNb].texI=gc.texShines[_Ti];\
  gc.shines[gc.shineNb].size=_Sz;\
  gc.shines[gc.shineNb].constant=_Cst;\
  A4DCp(_Col,gc.shines[gc.shineNb].color);\
  A4DMult(gc.shines[gc.shineNb].color,_In);\
  gc.shineNb++;

  SetShine(0, 40, 1, ColWhite, 1);
  SetShine(1, 20, 1, ColWhite, .5);
  SetShine(0 + f, 45, 0, ColWhite, 1);
  SetShine(1 + f, 45, 0, ColWhite, 1);
  SetShine(2 + f, 50, 0, ColWhite, 1);
  SetShine(3 + f, 55, 0, ColWhite, 1);
  SetShine(4 + f, 40, 0, ColWhite, 1);
  SetShine(5 + f, 35, 0, ColWhite, 1);
  SetShine(6 + f, 50, 0, ColWhite, 1);
  SetShine(7 + f, 40, 0, ColWhite, 1);
  SetShine(8 + f, 45, 0, ColWhite, 1);
  SetShine(9 + f, 50, 0, ColWhite, 1);
#undef SetShine
}


Module void
initTextures(void)
{
  UByte texImage[TexImageDim * TexImageDim * 4];
  GLenum gluerr;

  glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

  glGenTextures(1, &gc.texIds[TexGrid]);
  glBindTexture(GL_TEXTURE_2D, gc.texIds[TexGrid]);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  makeStripeImage(texImage, TexImageDim, TexImageDim);
  if ((gluerr = gluBuild2DMipmaps(GL_TEXTURE_2D, 3, TexImageDim, TexImageDim, GL_RGBA,
				GL_UNSIGNED_BYTE, (GLvoid *) (texImage)))) {
    fprintf(stderr, "GLULib%s\n", gluErrorString(gluerr));
    exit(-1);
  }


  glGenTextures(1, &gc.texIds[TexChecker]);
  glBindTexture(GL_TEXTURE_2D, gc.texIds[TexChecker]);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  makeCheckImage(texImage, TexImageDim, TexImageDim);
  if ((gluerr = gluBuild2DMipmaps(GL_TEXTURE_2D, 3, TexImageDim, TexImageDim, GL_RGBA,
				GL_UNSIGNED_BYTE, (GLvoid *) (texImage)))) {
    fprintf(stderr, "GLULib%s\n", gluErrorString(gluerr));
    exit(-1);
  }


  glGenTextures(1, &gc.texIds[TexPlate]);
  glBindTexture(GL_TEXTURE_2D, gc.texIds[TexPlate]);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
  glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  makeCircleImage(texImage, TexImageDim, TexImageDim);
  if ((gluerr = gluBuild2DMipmaps(GL_TEXTURE_2D, 3, TexImageDim, TexImageDim, GL_RGBA,
				GL_UNSIGNED_BYTE, (GLvoid *) (texImage)))) {
    fprintf(stderr, "GLULib%s\n", gluErrorString(gluerr));
    exit(-1);
  }


  loadImageTexture(TexPhoenix, "textures/phoenix.png", GL_RGB);
  loadImageTexture(TexFexp5, "textures/fexp5.png", GL_RGB);
  loadImageTexture(TexFrap, "textures/frap.png", GL_RGB);
  loadImageTexture(TexGeorge, "textures/george.png", GL_RGB);
  loadImageTexture(TexOGL, "textures/opengl.png", GL_LUMINANCE);

  gc.wallTextures[WallBack] = gc.texIds[TexFexp5];
  gc.wallTextures[WallLeft] = gc.texIds[TexPhoenix];
  gc.wallTextures[WallRight] = gc.texIds[TexFrap];
  gc.wallTextures[WallTop] = gc.texIds[TexGeorge];

  UAssert(texI < MaxTextureNb);
}

Module void
initGc(int *argc, char **argv)
{
  Int32 i;

  /*
   * Default settings 
   */

  bzero(&gc, sizeof(gc));
  gc.currentMode = 0;
  setPredefinedMode(gc.currentMode);

  /* window and perspective settings */
  gc.win = 0;
  gc.winOrigX = 0;
  gc.winOrigY = 0;
  gc.winWidth = 800;
  gc.winHeight = 600;
  gc.fullScreen = 0;
  gc.near = 0.1;
  gc.far = 6;

  /* rendering settings */
  A3DSet(gc.fogColor, .9, .9, 1.);
  //A3DSet(gc.fogColor, 0, 0, 0);
  A3DSet(gc.backgroundColor, .0, .0, .0);
  A4DSet(gc.helpColor, .3, 1., .3, 1);
  //gc.help = UFalse;
  gc.help = UTrue;		// FIXME release UTrue

  gc.boxSize = 1.2;
  gc.gridRes = 10;
  A4DSet(gc.gridColor, .4, .4, .4, 1);
  A4DSet(gc.pointColor, .1, 1, .1, 1);
  A4DSet(gc.wireColor, 1, 1, 1, 1);
  A4DSet(gc.normColor, .3, .3, .3, .2);
  A4DSet(gc.norm2Color, 1, .1, .3, 1);
  gc.wallResolution = WallResolution;

  /* texts */
  gc.helpTexts = helpTexts;
  gc.helpScale = .45;
  for (i = 0; i < HelpTextNb; i++) {
    gc.helpTextLen += strlen(gc.helpTexts[i]);
  }
  gc.statScale = .45;
  gc.bannerText = bannerText;
  gc.bannerScale = BannerScale;
  gc.bannerOffset = 10000;
  gc.bannerSpeed = BannerSpeed;
  gc.bannerColor1[0] = 20;
  gc.bannerColor1[1] = 20;
  gc.bannerColor1[2] = 255;
  gc.bannerColor1[3] = 255;
  gc.bannerColor2[0] = 220;
  gc.bannerColor2[1] = 20;
  gc.bannerColor2[2] = 20;
  gc.bannerColor2[3] = 255;

  /* stats */
  for (i = 0; i < FrameRateCount; i++) {
    DWChronoReset(&gc.frameChronos[i], DWChronoReal);
  }

  gc.mirrorTransp = 0;

  initLights();
  initMaterials();
  initObserver(&gc.obs);
  initSurfaces();
}

/* OpenGL settings 
 */
Module void
initOGL(void)
{
  glLightModelfv(GL_LIGHT_MODEL_LOCAL_VIEWER, gc.lModelLocal);
  glLightModelfv(GL_LIGHT_MODEL_TWO_SIDE, gc.lModelTwoSide);
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, gc.lModelAmbient);
  glEnable(GL_DEPTH_TEST);
  glDisable(GL_CULL_FACE);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glMatrixMode(GL_MODELVIEW);
  glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);
  glDisable(GL_AUTO_NORMAL);
  glDisable(GL_NORMALIZE);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  glGetDoublev(GL_MODELVIEW_MATRIX, gc.modelMat);
}

World int
main(int argc, char **argv)
{
  initGc(&argc, argv);

  glutInitWindowPosition(gc.winOrigX, gc.winOrigY);
  glutInitWindowSize(gc.winWidth, gc.winHeight);
  glutInit(&argc, argv);
  glutInitDisplayMode(GLUT_RGB | GLUT_DEPTH | GLUT_DOUBLE);

  if (!(gc.win = glutCreateWindow("OpenGL Ploter"))) {
    fprintf(stderr, "Error opening a window.\n");
    exit(-1);
  }
  reshapeWindow(gc.winWidth, gc.winHeight);
#ifdef XMESA
  XMesaSetFXmode(gc.fullScreen ? XMESA_FX_FULLSCREEN : XMESA_FX_WINDOW);
#endif

  glutKeyboardFunc(keyEventCB);
  glutSpecialFunc(specialEventCB);
  glutReshapeFunc(reshapeWindow);
  glutIdleFunc(drawScene);
  glutDisplayFunc(drawScene);

  initOGL();
  initFonts();
  initTextures();
  initFlares();

  glutMainLoop();

  return 0;
}
