#include "U.h"
#include "dwchrono.h"

/* this part is automaticaly updated, do NOT edit below */
/**************************************************************** Prototypes */

/* DWChronoSetCurrentTime
 *   Affect the current time to a variable
 * pt: pointer to this variable
 * ct: type of wanted time
 */
static void DWChronoSetCurrentTime(void *pt, DWChronoType ct);

/************************************************************ End Prototypes */
/* end of automaticaly updated part */



/* DWChronoSetCurrentTime
 *   Affect the current time to a variable
 * pt: pointer to this variable
 * ct: type of wanted time
 */
Module void
DWChronoSetCurrentTime(void *pt, DWChronoType ct)
{
  struct rusage rus;

  bzero(pt, sizeof(struct timeval));
  switch (ct) {
  case DWChronoReal:
    gettimeofday((struct timeval *) pt, NULL);
    break;
  case DWChronoUser:
    bzero(&rus, sizeof(struct rusage));		/* purify reports an UMR here... */
    getrusage(RUSAGE_SELF, &rus);
    *((struct timeval *) pt) = rus.ru_utime;
    break;
  case DWChronoSys:
    bzero(&rus, sizeof(struct rusage));		/* purify reports an UMR here... */
    getrusage(RUSAGE_SELF, &rus);
    *((struct timeval *) pt) = rus.ru_stime;
    break;
  default:
    DWNYI;
  }
}


/* DWReset
 *   Reset a chrono (set it to zero).
 * ct: chrono type (see DWChronoType) or 0 if you let it unchanged
 */
World void
DWChronoReset(DWChrono * pc, DWChronoType ct)
{
  DWAssert(pc);

  if (ct) {
    pc->ct = ct;
  }
  pc->cumulative = 0;
  pc->paused = DWFalse;
  DWChronoSetCurrentTime(&(pc->start), pc->ct);
}

/* DWChronoCreate
 *   Create (and start) a new chrono.
 * ct: chrono type (see DWChronoType)
 * return: created chrono (think to free it later)
 */
World DWChrono *
DWChronoCreate(DWChronoType ct)
{
  DWChrono *pc;

  pc = DWNew(DWChrono);
  DWChronoReset(pc, ct);
  return pc;
}


/* DWChronoPause
 *   Pause a chrono, ie: time stop to run till next Restart (or reset)
 * pc: chrono to be paused
 */
World void
DWChronoPause(DWChrono * pc)
{
  if (!pc->paused) {
    pc->cumulative = DWChronoGet(pc);
    pc->paused = DWTrue;
  }
}

/* DWChronoRestart
 *   Restart a previously paused chrono.
 * pc: chrono to be restarted
 */
World void
DWChronoRestart(DWChrono * pc)
{
  if (pc->paused) {
    pc->paused = DWFalse;
    DWChronoSetCurrentTime(&(pc->start), pc->ct);
  }
}


/* DWChronoGet
 *   Get elapsed time since last reset
 * pc: chrono to read from
 * return: number of milliseconds elapsed.
 */
World unsigned long
DWChronoGet(DWChrono * pc)
{
  struct timeval now;
  unsigned long dt;

  if (pc->paused)
    return pc->cumulative;

  DWChronoSetCurrentTime(&now, pc->ct);
  dt = ((unsigned long) now.tv_sec * 1000 + (now.tv_usec / 1000))
    - ((unsigned long) pc->start.tv_sec * 1000 + (pc->start.tv_usec / 1000));
  return pc->cumulative + dt;
}

/* DWMSPrint
 *   Print in a human readable format a milli second delay.
 *   (see also DWChronoPrint)
 * msec: delay to printA
 * str: if str is not NULL if will be filled else time will be printed by DWLPrintf
 * maxSize: size of str if not NULL
 * return:
 */
World char *
DWMSPrint(unsigned long msec, char *str, int maxSize)
{
  char buf[1024], *tmp;
  unsigned long sec, min, hour, day;
  int l;

  sec = msec / 1000UL;
  msec = msec % 1000UL;

  min = sec / 60;
  sec = sec % 60;

  hour = min / 60;
  min = min % 60;

  day = hour / 24;
  hour = hour % 24;

  tmp = buf;
  if (day) {
    sprintf(tmp, "%ldday%s ", day, (day > 1) ? "s" : "");
    tmp = buf + strlen(buf);
  }
  if (hour || day) {
    sprintf(tmp, "%ldh ", hour);
    tmp = buf + strlen(buf);
  }
  if (min || hour || day) {
    sprintf(tmp, "%ldmin ", min);
    tmp = buf + strlen(buf);
  }
  if (sec) {
    if (min || hour || day)
      sprintf(tmp, "%lds", sec);
    else
      sprintf(tmp, "%ld.%02lds", sec, msec);
    tmp = buf + strlen(buf);
  }
  else {
    sprintf(tmp, "%ldms", msec);
    tmp = buf + strlen(buf);
  }

  if ((maxSize == 0) || str == NULL) {
    printf(buf);
    return NULL;
  }
  else {
    l = Min(strlen(buf), maxSize - 1);
    strncpy(str, buf, l);
    str[l] = '\0';
    return str;
  }
}

/* DWChronoPrint
 *   Print a chrono in a human readable format.
 *   put human readable form in str assuming str is at least maxSize long
 *   if str is NULL (or maxSize == 0) print it on stdout instead of storing 
 *   it in str.
 * pc: chrono to print
 * str: where to put printed chorno
 * maxSize: size of str
 * return: str
 */
World void
DWChronoPrint(DWChrono * pc, char *str, int maxSize)
{
  char buf[1024], *tmp;
  int l;

  DWAssert(pc);

  tmp = buf;
  switch (pc->ct) {
  case DWChronoReal:
    sprintf(tmp, "Real: ");
    break;
  case DWChronoUser:
    sprintf(tmp, "User: ");
    break;
  case DWChronoSys:
    sprintf(tmp, "System: ");
    break;
  default:
    sprintf(tmp, "Unkown!: ");
    break;
  }
  tmp = buf + strlen(buf);

  DWMSPrint(DWChronoGet(pc), tmp, maxSize - strlen(buf));

  if ((maxSize == 0) || str == NULL) {
    printf(buf);
  }
  else {
    l = Min(strlen(buf), maxSize - 1);
    strncpy(str, buf, l);
    str[l] = '\0';
  }
}
