#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <errno.h>

#include "comm.h"
#include "util.h"

/* ?? TODO : make a better mem allocator for list (struct.c ) and RcvMsg cells */


/**************************************************************** Prototypes */

/* Read exactly msgLen bytes (do as many read as needed to fill pMsg).
 * Restart if interrupted by signals.
 *
 * Return: number of bytes read (as standard read).
 * or an error code <0 if some error occurs (broken pipe, peer socket...)
 */
static int UReadExactly(int fd, char *pMsg, int msgLen);

/* Read pending packets.
 *
 * Answer to internal request messages (like FLUSHREQ).
 * Add remaining messages to the queue.
 * If last message is truncated wait until completion.
 *
 * Return: 0 or error code
 */
static int UReadWaitingPackets(ConnectionLine * pConn);

/* Block until at least a new message is read
 * (no matter if some are already queued).
 * Don't flush neither ask for other line-end flush
 */
static int UWaitForNewMsg(ConnectionLine * pConn);

/* Return only if a message of type msgType is received (or already queued)
 * and return (remove from queue) this message thru ppMsg (as UGetNextMsgCommon)
 * don't flush neither ask for flush
 *
 * Return: 0 or error code
 *
 * Warning: Blocks until typed msg arrives...
 * but asks for nothing so may block for ever if nothing was asked before!
 */
static int UWaitForTypedMsg(ConnectionLine * pConn, MsgType msgType, char **ppMsg, int *pLen);

/* Make a round trip. ie ask for other line-end flush and wait until it's done
 * (round-trip done)
 * (no matter if some messages are already queued).
 * Flush this line-end too.
 */
static int UMakeRoundTrip(ConnectionLine * pConn);

/******************************************************************** Bodies */


/* Read exactly msgLen bytes (do as many read as needed to fill pMsg).
 * Restart if interrupted by signals.
 *
 * Return: number of bytes read (as standard read).
 * or an error code <0 if some error occurs (broken pipe, peer socket...)
 */
static int
UReadExactly(int fd, char *pMsg, int msgLen)
{
    int nbReadBytes, len;

    UTrace(TRACE_UCOMM, ("UReadExactly\n"));
    len = msgLen;
    while (msgLen) {
	while (((nbReadBytes = read(fd, pMsg, msgLen)) < 0) && (errno == EINTR));	/* empty */
	if (nbReadBytes == 0)
	    return UCOMM_ERR_DISCONNECTED;
	else if (nbReadBytes < 0)
	    URetErrno;
	else {
	    assert(nbReadBytes <= msgLen);
	    pMsg += nbReadBytes;
	    msgLen -= nbReadBytes;
	}
    }
    return len;
}

/* Read pending packets.
 *
 * Answer to internal request messages (like FLUSHREQ).
 * Add remaining messages to the queue.
 * If last message is truncated wait until completion.
 *
 * Return: 0 or error code
 */
static int
UReadWaitingPackets(ConnectionLine * pConn)
{
    fd_set readfds;
    struct timeval timeout;
    int nbReadBytes, msgLen;
    char header[sizeof(int) + sizeof(MsgType)], *pMsg;
    MsgType msgType;
    RcvMsg *pm;

    UTrace(TRACE_UCOMM, ("UReadWaitingPackets\n"));
    timeout.tv_sec = 0;
    timeout.tv_usec = 0;
    FD_ZERO(&readfds);
    /* read all pending packets (wait if truncated messages) */
    while (1) {
	FD_SET(pConn->socketNum, &readfds);
	if (URestartSelect(pConn->socketNum + 1,
			   (fd_set *) & readfds,
			   (fd_set *) NULL,
			   (fd_set *) NULL,
			   &timeout) < 0) {
	    URetErrno;
	}
	if (!FD_ISSET(pConn->socketNum, &readfds)) {
	    return 0;			/* Nothing is waiting to be read */
	}
	/* read length & type of next packet */
	if ((nbReadBytes = UReadExactly(pConn->socketNum, header, sizeof(int) + sizeof(MsgType))) < 0)
	    return nbReadBytes;

	/* get len and type of msg */
	msgLen = UIntBigEndianDec(*((int *) header));
	msgType = (MsgType) header[sizeof(int)];

#ifdef COMM_TRACE_MSG
	UConnPrint(pConn);
	fprintf(stderr, "reading a new message of len %d and type %s\n", msgLen, UMsgHeaderTypeToString(msgType));
#endif

	/* process msg depending on type */
	switch (msgType) {
	case UCOMM_MSG_REGISTERREQ:{
		Bools USCheckForGoodName(char *name);	/* beark hack ;) */

		/* app want to register it's name on this line */
		/* so this part is only done for server side */
		/* get name proposition */
		pMsg = (char *) UMalloc(sizeof(char) * msgLen);

		if (msgLen) {
		    if ((nbReadBytes = UReadExactly(pConn->socketNum, pMsg, msgLen)) < 0)
			return nbReadBytes;
		}
		/* check if name is already used */
		if (!USCheckForGoodName(pMsg)) {
		    /* bad name */
		    UFree(pMsg);
		    /* send bad name message and flush */
		    URetIfNz(USendCommon(pConn, UCOMM_MSG_REGISTERBAD, NULL, 0));
		    URetIfNz(UFlushCommon(pConn));
		}
		else {
		    pConn->lineName = pMsg;
		    /* send ack and flush */
		    URetIfNz(USendCommon(pConn, UCOMM_MSG_REGISTERACK, NULL, 0));
		    URetIfNz(UFlushCommon(pConn));
		}
		return 0;
		break;
	    }
	case UCOMM_MSG_REGISTERBAD:{
		/* app received 'register failed' so raise an error */
		return UCOMM_ERR_BADNAME;
		break;
	    }
	case UCOMM_MSG_FLUSHREQ:{	/* other line-end ask for a flush */
		/* send ack and flush */
		URetIfNz(USendCommon(pConn, UCOMM_MSG_FLUSHACK, NULL, 0));
		URetIfNz(UFlushCommon(pConn));
		return 0;
		break;
	    }
	case UCOMM_MSG_REGISTERACK:	/* app received ack so register done
					 * go on */
	case UCOMM_MSG_FLUSHACK:	/* flush has been done by other
					 * line-end */
	    /*
	     * left acks &co in queue (they are removed by other functions
	     * before user may see them
	     */
	case UCOMM_MSG_USER:{
		pMsg = (char *) UMalloc(sizeof(char) * msgLen + 1);

		if (msgLen) {
		    if ((nbReadBytes = UReadExactly(pConn->socketNum, pMsg, msgLen)) < 0)
			return nbReadBytes;
		}
		pMsg[msgLen] = '\0';	/* easier to read for debug ;) */
		/* store this new msg in queue */
		pm = UNew(RcvMsg);
		pm->msgType = msgType;
		pm->msgLen = msgLen;
		pm->pMsg = pMsg;
#ifdef COMM_TRACE_MSG
		printf("got message ->");
		UPrintAll(pMsg);	/* ?? DEBUG only */
		printf("<-\n");
#endif
		lAddLast(pConn->lReceivedMsg, pm);	/* TODO ?? is lAddLast
							 * optimized??? */
		pConn->nbReceivedMsg++;
		return 0;
		break;
	    }
	default:{
		UIError("UReadWaitingPackets", "msgType %d unknown", msgType);
	    }
	}
    }
}

/* Incorporate waiting packets.
 *
 * Read pending packets.
 * Answer to internal request messages (like FLUSHREQ).
 * Add remaining messages to the queue.
 * If last message is truncated wait until completion.
 *
 * Return: 0 or error code
 */
int
UIncWaitingPacketsCommon(ConnectionLine * pConn)
{
    return UReadWaitingPackets(pConn);
}

/* Extract msg from a struct RcvMsg
 */
char *
UKeyRcvMsg(RcvMsg * pm)
{
    return pm->pMsg;
}

/* Free a struct RcvMsg
 */
void
UFreeRcvMsg(RcvMsg * pm)
{
    UFree(pm->pMsg);
    UFree(pm);
}

/* Print a struct RcvMsg (debug)
 */
void
UPrintRcvMsg(RcvMsg * pm)
{
    printf("type:%s len:%d ->", UMsgHeaderTypeToString(pm->msgType), pm->msgLen);
    UDumpMem(pm->pMsg, pm->msgLen, UDUMP_ASC_OR_DEC);
    printf("<-");

}


/* Print some info on this connectionLine
 * Useful for debug
 */
void
UConnPrint(ConnectionLine * pConn)
{
    printf("(lineName:%s socketNum:%d packToSendLen:%d nbReceivedMsg:%d)\n",
	   pConn->lineName, pConn->socketNum, pConn->packToSendLen, pConn->nbReceivedMsg);
}

/* Add a connectionLine (in a GArray of)
 */
int
UAddAConn(ConnectionLine * pConn, GArray ** paConn)
{
    UTrace(TRACE_UCOMM, ("UAddAConn\n"));
    if (!*paConn) {
	*paConn = gaCreate(UConnPrint);
    }
    return gaAdd(*paConn, pConn);
}

/* Send a message (or at least store it in a buffer).
 *
 * Return: 0 or error code
 */
int
USendCommon(ConnectionLine * pConn /* On which line to send msg */ ,
	    MsgType msgType /* Type of msg */ ,
	    void *pMsg /* Pointer on data to send */ ,
	    int len /* Lenght of data to send */ )
{
    char *endPacket, *pTailMsg;
    int msgLen, encMsgLen, tailMsgLen, copyLen, totLen;

    UTrace(TRACE_UCOMM, ("USendCommon\n"));
#ifdef COMM_TRACE_MSG
    UConnPrint(pConn);
    fprintf(stderr, "sending a new message of len %d and type %s\n", len, UMsgHeaderTypeToString(msgType));
    printf("sent message ->");
    UPrintAll(pMsg);			/* ?? DEBUG only */
    printf("<-\n");
#endif
    msgLen = len;

    /* assert that header of a message is not cut off message */
    /* assert that short messages aren't split in two packets */
    /* avoid complex manipulations but doesn't */
    /* increase too much number of send packets (system calls) */
    totLen = len + sizeof(int) + sizeof(MsgType);
    if (((sizeof(int) + sizeof(MsgType)) > (UCOMM_PACKET_SIZE - pConn->packToSendLen)) ||
	((totLen <= UCOMM_PACKET_SIZE) &&
	 (totLen > (UCOMM_PACKET_SIZE - pConn->packToSendLen))))
	URetIfNz(UFlushCommon(pConn));

    /* put len encoded */
    endPacket = pConn->packToSend + pConn->packToSendLen;
    encMsgLen = UIntBigEndianEnc(msgLen);	/* encode len */
    bcopy(&encMsgLen, endPacket, sizeof(int));
    endPacket += sizeof(int);
    pConn->packToSendLen += sizeof(int);

    /* put type */
    bcopy(&msgType, endPacket, sizeof(MsgType));
    endPacket += sizeof(MsgType);
    pConn->packToSendLen += sizeof(MsgType);

    /* check if packet is full */
    if ((pConn->packToSendLen) == UCOMM_PACKET_SIZE) {
	URetIfNz(UFlushCommon(pConn));
	endPacket = pConn->packToSend;
    }

    /* put msg */
    pTailMsg = pMsg;
    tailMsgLen = len;
    while (tailMsgLen) {
	copyLen = Min(tailMsgLen, UCOMM_PACKET_SIZE - pConn->packToSendLen);
	bcopy(pTailMsg, endPacket, copyLen);
	pTailMsg += copyLen;
	tailMsgLen -= copyLen;
	if ((pConn->packToSendLen += copyLen) == UCOMM_PACKET_SIZE) {
	    URetIfNz(UFlushCommon(pConn));
	    endPacket = pConn->packToSend;
	}
    }

    /* check if enough room for next msg len and type */
    if ((sizeof(int) + sizeof(MsgType)) > (UCOMM_PACKET_SIZE - pConn->packToSendLen)) {
	URetIfNz(UFlushCommon(pConn));
	endPacket = pConn->packToSend;
    }
    pConn->nbSendMsg++;
    URetIfNz(UReadWaitingPackets(pConn));	/* inc waiting packets */
    return 0;
}

/* Send all messages which may have been buffered (if any)
 * (ie flush actual packet (if needed)).
 *
 * Return: 0 or error code
 */
int
UFlushCommon(ConnectionLine * pConn)
{
    int nbWBytes;

    UTrace(TRACE_UCOMM, ("UFlushCommon\n"));
    if (pConn->packToSendLen) {
	if ((nbWBytes = write(pConn->socketNum, pConn->packToSend, pConn->packToSendLen)) < 0) {
	    URetErrno;
	}
	assert(nbWBytes == pConn->packToSendLen);
#ifdef COMM_TRACE_MSG
	printf("flush len %d ->%s<-\n", pConn->packToSendLen, UDumpMem(pConn->packToSend, pConn->packToSendLen, UDUMP_ASC_OR_DEC));
#endif
	bzero(pConn->packToSend, UCOMM_PACKET_SIZE);	/* ?? for debug only */
	pConn->packToSendLen = 0;
	pConn->nbSendPacket++;
    }
    URetIfNz(UReadWaitingPackets(pConn));	/* inc waiting packets */
    return 0;
}

/* How many messages are waiting for a read.
 *
 * Return imediatly if messages are already queued.
 * If no message queued, flush buffer and wait for roundtrip
 * (ie be sure no message are kept in other line end buffer)
 *
 * Warning: May block a bit if correspondant is too busy.
 */
int
UPendingMsgCommon(ConnectionLine * pConn, int *pNbMsg)
{

    UTrace(TRACE_UCOMM, ("UPendingMsgCommon\n"));
    URetIfNz(UReadWaitingPackets(pConn));	/* inc waiting packets */
    if (!(*pNbMsg = lNbElts(pConn->lReceivedMsg))) {
	URetIfNz(UMakeRoundTrip(pConn));
	*pNbMsg = lNbElts(pConn->lReceivedMsg);
    }
    return 0;
}

/* How many messages are waiting for a read.
 *
 * Checks for waiting packets and return
 * nether blocks
 */
int
UCheckPendingMsgCommon(ConnectionLine * pConn, int *pNbMsg)
{
    UTrace(TRACE_UCOMM, ("UCheckPendingMsgCommon\n"));
    URetIfNz(UReadWaitingPackets(pConn));	/* inc waiting packets */
    *pNbMsg = lNbElts(pConn->lReceivedMsg);
    return 0;
}

/* Checks if there is something waiting for a read on all connected lines.
 *
 * Do the check as fast as possible (only one select)
 * nether blocks
 */
int
UCheckAllLinesCommon(GArray * aConn, int *pIsMsg)
{
    int nbe, i, maxSock = 0;
    ConnectionLine *pConn;
    fd_set readfds;
    struct timeval timeout;

    UTrace(TRACE_UCOMM, ("UCheckAllLinesCommon\n"));
    *pIsMsg = 0;
    FD_ZERO(&readfds);
    nbe = gaSize(aConn);
    for (i = 0; i < nbe; i++) {
	pConn = gaLookNth(aConn, i);
	if (pConn) {
	    if (lNbElts(pConn->lReceivedMsg)) {
		/* there's already something in the queue */
		*pIsMsg = 1;
		return 0;
	    }
	    FD_SET(pConn->socketNum, &readfds);
	    maxSock = Max(maxSock, pConn->socketNum);
	}
    }
    timeout.tv_sec = 0;
    timeout.tv_usec = 0;
    if (URestartSelect(maxSock + 1,
		       (fd_set *) & readfds,
		       (fd_set *) NULL,
		       (fd_set *) NULL,
		       &timeout) < 0) {
	URetErrno;
    }
    if (FD_ISSET(maxSock, &readfds))
	*pIsMsg = 1;			/* there's something on one of the
					 * connected lines */
    return 0;
}


/* Fill a fd_set (eg: for a select) with sockets fd of all limes stored
 * in aConn
 *
 * Return: max fd stored in *pFdSet
 */
int
UFillFdAllLinesCommon(GArray * aConn, fd_set * pFdSet)
{
    int nbe, i, maxSock = 0;
    ConnectionLine *pConn;

    UTrace(TRACE_UCOMM, ("UFillFdAllLinesCommon\n"));
    if (!aConn)
	return 0;
    nbe = gaSize(aConn);
    for (i = 0; i < nbe; i++) {
	pConn = gaLookNth(aConn, i);
	if (pConn) {
	    FD_SET(pConn->socketNum, pFdSet);
	    maxSock = Max(maxSock, pConn->socketNum);
	}
    }
    return maxSock;
}


/* Get next message.
 *
 * If no message in waiting queue;
 * Flush this line-end.
 * Make periodics roundtrip to insure no messages are in buffer at the
 * other line-end
 *
 * Return: 0 or error code.
 *
 * Warning: It's a blocking call.
 */
int
UGetNextMsgCommon(ConnectionLine * pConn,
		  char **ppMsg		/* Pointer on returned message
		      (caller must free ppMsg after use) */ ,
		  int *pLen /* Lenght of returned message */ )
{
    RcvMsg *pm;

    UTrace(TRACE_UCOMM, ("UGetNextMsgCommon\n"));
    if (ppMsg)
	*ppMsg = NULL;
    if (pLen)
	*pLen = 0;
    URetIfNz(UWaitForMsgCommon(pConn));	/* inc waiting packets */
    if (lNbElts(pConn->lReceivedMsg)) {
	pm = lGetHead(pConn->lReceivedMsg);
	if (ppMsg)
	    *ppMsg = pm->pMsg;
	else
	    UFree(pm->pMsg);
	if (pLen)
	    *pLen = pm->msgLen;
	UFree(pm);
    }
    return 0;
}

/* Peek next message (message is not removed from queue).
 *
 * If no message in waiting queue;
 * Flush this line-end.
 * Make periodics roundtrip to insure no messages are in buffer at the
 * other line-end.
 *
 * Return: 0 or error code.
 *
 * Warning:  It's a blocking call.
 */
int
UPeekNextMsgCommon(ConnectionLine * pConn,
		   char **ppMsg		/* Pointer on returned message
					 * (caller must NOT free ppMsg after
		       use) */ ,
		   int *pLen /* Lenght of returned message */ )
{
    RcvMsg *pm;

    UTrace(TRACE_UCOMM, ("UPeekNextMsgCommon\n"));
    if (ppMsg)
	*ppMsg = NULL;
    if (pLen)
	*pLen = 0;
    URetIfNz(UWaitForMsgCommon(pConn));	/* inc waiting packets */
    if (lNbElts(pConn->lReceivedMsg)) {
	pm = lLookHead(pConn->lReceivedMsg);
	if (ppMsg)
	    *ppMsg = pm->pMsg;
	if (pLen)
	    *pLen = pm->msgLen;
    }
    return 0;
}

/* Return (and remove) first matching message from queue.
 *
 * Make no round trip so only messages which are already waiting are searched
 * (incorporate waiting packets before search)
 */
int
UGetFirstMatchingMsgCommon(ConnectionLine * pConn,
			   char *head	/* String which must be the head of
			       matching message */ ,
			   char **ppMsg,
			   int *pLen)
{
    int nbm, i;
    RcvMsg *pm;

    UTrace(TRACE_UCOMM, ("UGetFirstMatchingMsgCommon\n"));
    if (ppMsg)
	*ppMsg = NULL;
    if (pLen)
	*pLen = 0;
    URetIfNz(UReadWaitingPackets(pConn));	/* inc waiting packets */
    nbm = lNbElts(pConn->lReceivedMsg);
    for (i = 0; i < nbm; i++) {
	pm = lLookNth(pConn->lReceivedMsg, i);
	if (!strcmp(head, pm->pMsg)) {
	    if (ppMsg)
		*ppMsg = pm->pMsg;
	    else
		UFree(pm->pMsg);
	    if (pLen)
		*pLen = pm->msgLen;
	    lDel(pConn->lReceivedMsg, pm, False);
	    UFree(pm);
	    return 0;
	}
    }
    return 0;
}

/* Return (but don't remove) first matching message form queue.
 *
 * Make no round trip so only messages which are already waiting are searched
 * (incorporate waiting packets before search)
 */
int
UPeekFirstMatchingMsgCommon(ConnectionLine * pConn,
			    char *head	/* String which must be the head of
			        matching message */ ,
			    char **ppMsg,
			    int *pLen)
{
    int nbm, i;
    RcvMsg *pm;

    if (ppMsg)
	*ppMsg = NULL;
    if (pLen)
	*pLen = 0;
    URetIfNz(UReadWaitingPackets(pConn));	/* inc waiting packets */
    nbm = lNbElts(pConn->lReceivedMsg);
    for (i = 0; i < nbm; i++) {
	pm = lLookNth(pConn->lReceivedMsg, i);
	if (!strcmp(head, pm->pMsg)) {
	    if (ppMsg)
		*ppMsg = pm->pMsg;
	    if (pLen)
		*pLen = pm->msgLen;
	    return 0;
	}
    }
    return 0;
}

/* Block until at least a new message is read
 * (no matter if some are already queued).
 * Don't flush neither ask for other line-end flush
 */
static int
UWaitForNewMsg(ConnectionLine * pConn)
{
    int nbMsg;

    UTrace(TRACE_UCOMM, ("UWaitForNewMsg\n"));
    nbMsg = lNbElts(pConn->lReceivedMsg);
    while (lNbElts(pConn->lReceivedMsg) == nbMsg) {
	URetIfNz(UReadWaitingPackets(pConn));	/* waiting packets */
	UUsleep(UCOMM_LOOP_DELAY);
    }
    return 0;
}

/* Return only if a message of type msgType is received (or already queued)
 * and return (remove from queue) this message thru ppMsg (as UGetNextMsgCommon)
 * don't flush neither ask for flush
 *
 * Return: 0 or error code
 *
 * Warning: Blocks until typed msg arrives...
 * but asks for nothing so may block for ever if nothing was asked before!
 */
static int
UWaitForTypedMsg(ConnectionLine * pConn, MsgType msgType, char **ppMsg, int *pLen)
{
    int nbm, i;
    RcvMsg *pm;

    UTrace(TRACE_UCOMM, ("UWaitForTypedMsg\n"));
    if (ppMsg)
	*ppMsg = NULL;
    if (pLen)
	*pLen = 0;
    URetIfNz(UReadWaitingPackets(pConn));	/* read waiting packets */
    while (1) {
	nbm = lNbElts(pConn->lReceivedMsg);
	for (i = 0; i < nbm; i++) {	/* search msg of desired type */
	    pm = lLookNth(pConn->lReceivedMsg, i);
	    if (msgType == pm->msgType) {
		if (ppMsg)
		    *ppMsg = pm->pMsg;
		else
		    UFree(pm->pMsg);
		if (pLen)
		    *pLen = pm->msgLen;
		lDel(pConn->lReceivedMsg, pm, False);
		UFree(pm);
		return 0;
	    }
	}
	URetIfNz(UWaitForNewMsg(pConn));
    }
    return 0;
}




/* Make a round trip. ie ask for other line-end flush and wait until it's done
 * (round-trip done)
 * (no matter if some messages are already queued).
 * Flush this line-end too.
 */
static int
UMakeRoundTrip(ConnectionLine * pConn)
{
    UTrace(TRACE_UCOMM, ("UMakeRoundTrip\n"));
    URetIfNz(USendCommon(pConn, UCOMM_MSG_FLUSHREQ, NULL, 0));
    URetIfNz(UFlushCommon(pConn));
    URetIfNz(UWaitForTypedMsg(pConn, UCOMM_MSG_FLUSHACK, NULL, NULL));
    return 0;
}

/* Return only if a message is received (or already queued).
 *
 * Flush if needed and ask periodicaly for other line-end flush until
 * message is received.
 *
 * Return: 0 or error code
 */
int
UWaitForMsgCommon(ConnectionLine * pConn)
{
    UTrace(TRACE_UCOMM, ("UWaitForMsgCommon\n"));
    if (lNbElts(pConn->lReceivedMsg))
	return 0;
    UFlushCommon(pConn);		/* flush this */
    while (!lNbElts(pConn->lReceivedMsg)) {
	URetIfNz(UReadWaitingPackets(pConn));	/* incorporate waiting
						 * packets */
	if (lNbElts(pConn->lReceivedMsg))
	    return 0;
	URetIfNz(UMakeRoundTrip(pConn));/* flush other line-end */
	if (lNbElts(pConn->lReceivedMsg))
	    return 0;
	UUsleep(UCOMM_LOOP_DELAY);	/* wait a bit */
    }
    return 0;
}




/* Return id associated to line nammed (lineName) in aConn
 * or -1 if no such line
 */
int
UNameToIdCommon(char *lineName, GArray * aConn)
{
    int i, nbc;
    ConnectionLine *pConn;

    UTrace(TRACE_UCOMM, ("UNameToIdCommon\n"));
    nbc = gaSize(aConn);
    for (i = 0; i < nbc; i++) {
	pConn = gaLookNth(aConn, i);
	if (pConn) {
	    if (!strcmp(pConn->lineName, lineName))
		return i;
	}
    }
    return -1;
}

/* Return name of line identified by id in aConn
 * or NULL if no such line
 * caller is responsible for freeing the returned string
 */
char *
UIdToNameCommon(int id, GArray * aConn)
{
    ConnectionLine *pConn;

    UTrace(TRACE_UCOMM, ("UIdToNameCommon\n"));
    if ((pConn = gaLookNth(aConn, id))) {
	return UStrDup(pConn->lineName);
    }
    return (char *) NULL;
}

/* Given a id (and correct array of pConn) find its fd (socket number here)
 * useful when need to do a select
 * return fd number or -1 if no such id
 */
int
UIdToFdCommon(int id, GArray * aConn)
{
    ConnectionLine *pConn;

    UTrace(TRACE_UCOMM, ("ULineToFdCommon\n"));
    if ((pConn = gaLookNth(aConn, id))) {
	return pConn->socketNum;
    }
    return -1;
}


/* Close a line in aConn.
 *
 * Remember to close all lines before end of
 * application or sockets won't be freed until some minutes
 *
 * Warning: any pending message on this line will be lost
 */
int
UCloseLineCommon(int id, GArray * aConn)
{
    ConnectionLine *pConn;

    UTrace(TRACE_UCOMM, ("UCloseLineCommon\n"));
    pConn = gaLookNth(aConn, id);
    close(pConn->socketNum);
    if (pConn) {
	gaDelNth(aConn, id, False);
	UFree(pConn->lineName);
	lFree(pConn->lReceivedMsg, True);
	UFree(pConn);
    }
    return 0;
}

/* Close all lines in aConn.
 *
 * Remember to close all lines before end of
 * application or sockets won't be freed until some minutes
 */
int
UCloseAllLinesCommon(GArray * aConn)
{
    int i, nbc;

    UTrace(TRACE_UCOMM, ("UCloseAllLinesCommon\n"));
    nbc = gaSize(aConn);
    for (i = 0; i < nbc; i++) {
	UCloseLineCommon(i, aConn);
    }
    return 0;
}

/* Convert an error code to a string.
 *
 * Caller is responsible for freeing the returned string
 */
char *
UErrCodeToErrString(int errCode)
{
    char tmp[MAX_CHAR_TMP];
    extern char *sys_errlist[];

    if (errCode > 0) {
	sprintf(tmp, "Error: (%d) %s ", errCode, sys_errlist[errCode]);
    }
    else {
	switch (errCode) {
	case UCOMM_ERR_GETHOSTBYNAME:
	    sprintf(tmp, "Error: (%d) bad hostname ", errCode);
	    break;
	case UCOMM_ERR_CONNECT:
	    sprintf(tmp, "Error: (%d) can't connect ", errCode);
	    break;
	case UCOMM_ERR_DISCONNECTED:
	    sprintf(tmp, "Error: (%d) line disconnected (other line-end probably dead) ", errCode);
	    break;
	default:
	    UIError("UErrCodeToErrString", "Error: ?? unknown errCode (%d)", errCode);
	}
    }
    return UStrDup(tmp);
}

/* Convert a MsgType to String
 */
char *
UMsgHeaderTypeToString(MsgType type)
{
    switch (type) {
	case UCOMM_MSG_USER:return "user";
    case UCOMM_MSG_REGISTERREQ:
	return "register Req";
    case UCOMM_MSG_REGISTERACK:
	return "register Ack";
    case UCOMM_MSG_REGISTERBAD:
	return "register Bad";
    case UCOMM_MSG_FLUSHREQ:
	return "flush Req";
    case UCOMM_MSG_FLUSHACK:
	return "flush Ack";
    default:
	return "UNKNOWN!!!";
    }
}

/* Register name for this line (only for apps).
 *
 * Return: 0 or error code (UCOMM_ERR_BADNAME for a bad name)
 */
int
URegisterNameAppSide(int id, GArray * aConn, char *lineName)
{
    ConnectionLine *pConn;

    UTrace(TRACE_UCOMM, ("URegisterNameAppSide\n"));
    pConn = gaLookNth(aConn, id);
    URetIfNz(USendCommon(pConn, UCOMM_MSG_REGISTERREQ, lineName, strlen(lineName) + 1));
    URetIfNz(UFlushCommon(pConn));
    URetIfNz(UWaitForTypedMsg(pConn, UCOMM_MSG_REGISTERACK, NULL, NULL));
    return 0;
}
