#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/time.h>
#include <string.h>
#include <signal.h>

#include "comm.h"
#include "util.h"


static GArray *aAppConn = NULL;		/* array of ConnectionLines */


/**************************************************************** Prototypes */


/******************************************************************** Bodies */


/* Opens a line of connection from this application to a server.
 *
 * The server is defined by it's address and a port number.
 * An id is returned which identify this line. So message communications on
 * this line must use this id.
 * If a less crude identification is needed by server this line may be named
 * (with a string). Use URegisterName to do so.
 *
 * Return: 0 or an error code
 *
 * Warning: this function set the signal handler for SIG_PIPE to SIG_IGN
 */
int
UAOpenLine(char *serverAdress, int portNum, int *pId)
{
    static Bools handlerSet = False;
    struct hostent *adress;
    struct sockaddr_in server;
    int socketNum;
    ConnectionLine *pConn;

    UTraceSetLevel(TRACE_UCOMM);
    if (!handlerSet) {
	handlerSet = True;
	signal(SIGPIPE, SIG_IGN);
    }
    if ((socketNum = socket(AF_INET, SOCK_STREAM, 0)) < 0)
	return errno;

    if ((adress = gethostbyname(serverAdress)) == NULL) {	/* get server adress */
	return UCOMM_ERR_GETHOSTBYNAME;
    }

    server.sin_family = AF_INET;	/* Using Internet domain */
    server.sin_port = htons(portNum);
    bcopy((char *) adress->h_addr, (char *) &server.sin_addr, adress->h_length);

    if ((connect(socketNum, (struct sockaddr *) & server, sizeof(server))) < 0) {
	close(socketNum);		/* The server is not connected */
	return UCOMM_ERR_CONNECT;
    }

    /* it seem's ok so add this new line in ConnectionLines array aAppConns */
    pConn = UNew(ConnectionLine);
    pConn->lineName = NULL;
    pConn->socketNum = socketNum;
    pConn->lReceivedMsg = lCreate((void *(*) ()) UKeyRcvMsg, strcmp, UFreeRcvMsg, UPrintRcvMsg);
    bzero(pConn->packToSend, UCOMM_PACKET_SIZE);
    pConn->packToSendLen = 0;
    pConn->nbSendPacket = 0;
    pConn->nbSendMsg = 0;
    pConn->nbReceivedMsg = 0;
    *pId = UAddAConn(pConn, &aAppConn);
    return 0;
}

/* Register name for this line.
 *
 * Return: 0 or error code (UCOMM_ERR_BADNAME for a bad name)
 */
int
UARegisterName(int id, char *lineName)
{
    return URegisterNameAppSide(id, aAppConn, lineName);
}

/* Close a line.
 *
 * Remember to close all lines before end of
 * application or sockets won't be freed until some minutes
 *
 * Warning: any pending message on this line will be lost
 */
int
UACloseLine(int id)
{
    return UCloseLineCommon(id, aAppConn);
}

/* Close all lines.
 *
 * Remember to close all lines before end of
 * application or sockets won't be freed until some minutes
 */
int
UACloseAllLines(void)
{
    return UCloseAllLinesCommon(aAppConn);
}

/* Send a message (on at least store it in a buffer).
 *
 * Return: 0 or error code
 */
int
UASend(int id /* On which line to send msg */ ,
       void *msg /* Pointer on data to send */ ,
       int len /* Lenght of data to send */ )
{
    return USendCommon((ConnectionLine *) gaLookNth(aAppConn, id), UCOMM_MSG_USER, msg, len);
}

/* Send all messages which may have been buffered (if any)
 * (ie flush actual packet (if needed))
 *
 * Return: 0 or error code
 */
int
UAFlush(int id)
{
    return UFlushCommon((ConnectionLine *) gaLookNth(aAppConn, id));
}

/* How many messages are waiting for a read.
 *
 * Return imediatly if messages are already queued.
 * If no message queued, flush buffer and wait for roundtrip
 * (ie be sure no message are kept in other line end buffer).
 *
 * Warning: so may block a bit if correspondant is too busy.
 */
int
UAPendingMsg(int id, int *pNbMsg)
{
    return UPendingMsgCommon((ConnectionLine *) gaLookNth(aAppConn, id), pNbMsg);
}


/* How many messages are waiting for a read.
 *
 * Checks for waiting packets abd return
 * nether blocks
 */
int
UACheckPendingMsg(int id, int *pNbMsg)
{
    return UCheckPendingMsgCommon((ConnectionLine *) gaLookNth(aAppConn, id), pNbMsg);
}

/* Checks if there is something waiting for a read on all connected lines.
 *
 * Do the check as fast as possible (only one select)
 * nether blocks
 */
int
UACheckAllLines(int *pIsMsg)
{
    return UCheckAllLinesCommon(aAppConn, pIsMsg);
}

/* Get next message
 *
 * If no message in waiting queue :
 * - flush this line-end
 * - make periodics roundtrip to insure no messages are in buffer at the
 *   other line-end
 *
 * Return: 0 or error code
 *
 * Warning :  It's a blocking call.
 */
int
UAGetNextMsg(int id,
	     char **ppMsg		/* Pointer on returned message
					 * (caller must free ppMsg after
	         use). */ ,
	     int *pLen /* Lenght of returned message. */ )
{
    return UGetNextMsgCommon((ConnectionLine *) gaLookNth(aAppConn, id), ppMsg, pLen);
}

/* Peek next message (message is not removed from queue).
 *
 * if no message in waiting queue :
 * - flush this line-end
 * - make periodics roundtrip to insure no messages are in buffer at the
 *   other line-end
 *
 * Return: 0 or error code
 *
 * Warning: It's a blocking call.
 */
int
UAPeekNextMsg(int id,
	      char **ppMsg		/* Pointer on returned message
					 * (caller must NOT free ppMsg after
	          use). */ ,
	      int *pLen /* Lenght of returned message */ )
{
    return UPeekNextMsgCommon((ConnectionLine *) gaLookNth(aAppConn, id), ppMsg, pLen);
}

/* Return (and remove) first matching message from queue.
 *
 * Make no round trip so only messages which are already waiting are searched
 * (incorporate waiting packets before search)
 */
int
UAGetFirstMatchingMsg(int id,
		      char *head	/* String which must be the head of
		          matching message */ ,
		      char **ppMsg,
		      int *pLen)
{
    return UGetFirstMatchingMsgCommon((ConnectionLine *) gaLookNth(aAppConn, id), head, ppMsg, pLen);
}

/* Return (but don't remove) first matching message form queue.
 *
 * Make no round trip so only messages which are already waiting are searched
 * (incorporate waiting packets before search)
 * head: string which must be the head of matching message
 */
int
UAPeekFirstMatchingMsg(int id,
		       char *head	/* String which must be the head of
		           matching message */ ,
		       char **ppMsg,
		       int *pLen)
{
    return UPeekFirstMatchingMsgCommon((ConnectionLine *) gaLookNth(aAppConn, id), head, ppMsg, pLen);
}

/* Return id associated to line nammed (lineName)reg
 * or -1 if no such line
 */
int
UANameToId(char *lineName)
{
    return UNameToIdCommon(lineName, aAppConn);
}

/* Return name of line identified by id
 * or NULL if no such line
 *
 * Caller is responsible for freeing the returned string
 */
char *
UAIdToName(int id)
{
    return UIdToNameCommon(id, aAppConn);
}


/* Given a id return its fd (socket number here) 
 * useful when need to do a select 
 * return fd number or -1 if no such id
 */
int
UAIdToFd(int id /* line id */){
    return UIdToFdCommon(id,aAppConn);
}

/* Convert an error code to a string
 *
 * caller is responsible for freeing the returned string
 */
char *
UAErrCodeToErrString(int errCode)
{
    return UErrCodeToErrString(errCode);
}
