#include <string.h>
#include <memory.h>
#include <malloc.h>
#include <limits.h>

#include "util.h"


/**************************************************************** Prototypes */

/* ODOT...
 */
static void
     lFreeListElt(List * pl, ListElt * ple, Bools freeElt);

/* ODOT...
 */
static ListElt *
        lLookListElt(List * pl, void *pe);

/* Return: First ListElt* in which pe match key pk.
 */
static ListElt *
        lLookListKey(List * pl, void *pk);

/******************************************************************** Bodies */



/*****************************************************************************\
*                                                                             *
* List                                                                        *
*                                                                             *
\*****************************************************************************/

/* ODOT...
 */
List *
     lCreate(void *(*keyFunc) (), int (*orderFunc) (), void (*freeFunc) (),
	     void (*printFunc) ())
{
    List *pl;
    static number = 1;

    if ((keyFunc == NULL) || (orderFunc == NULL) || (freeFunc == NULL))
	UIError("createList", "bad parameters");
    pl = UNew(List);
    pl->nbe = 0;
    pl->number = number++;
    pl->modified = True;
    pl->lastN = 0;
    pl->lastAns = NULL;
    pl->keyFunc = keyFunc;
    pl->ordFunc = orderFunc;
    pl->freeFunc = freeFunc;
    if (printFunc == NULL)
	printFunc = printVoid;
    pl->printFunc = printFunc;
    pl->pFirst = NULL;
    pl->pLast = NULL;
    return pl;
}

/* ODOT...
 */
static void
lFreeListElt(List * pl, ListElt * ple, Bools freeElt)
{
    if (freeElt) {
	(*(pl->freeFunc)) (ple->pe);
    }
    free(ple);
}

/* ODOT...
 */
void
lFree(List * pl, Bools freeElt)
{
    ListElt *ple, *pleb;

    ple = pl->pFirst;
    while (ple) {
	pleb = ple;
	ple = ple->next;
	lFreeListElt(pl, pleb, freeElt);
    }
    free(pl);
}

/* Add pe in first place of list pl.
 */
void
lAdd(List * pl, void *pe)
{
    ListElt *ple;

    pl->modified = True;
    ple = (ListElt *) UMallocMsg(sizeof(ListElt), "alloc a new list element ");
    ple->pe = pe;
    ple->flag = 0;
    ple->next = pl->pFirst;
    pl->pFirst = ple;
    if (!pl->pLast)
	pl->pLast = ple;
    pl->nbe++;
}

/* Add pe in first place of list pl
 * but only if pe is not already present (tested with ordFunc)
 */
void
lAddUnique(List * pl, void *pe)
{
    /* ?? to be optimized */

    if (lIn(pl, pe) == False)
	lAdd(pl, pe);
}

/* Add pe in last place of list pl
 */
void
lAddLast(List * pl, void *pe)
{
    ListElt *ple;

    pl->modified = True;
    ple = (ListElt *) UMallocMsg(sizeof(ListElt), "alloc a new list element");
    ple->pe = pe;
    ple->flag = 0;
    ple->next = NULL;
    if (pl->pLast) {
	pl->pLast->next = ple;
	pl->pLast = ple;
    }
    else				/* empty list */
	pl->pFirst = pl->pLast = ple;
    pl->nbe++;
}

/* ODOT...
 */
int
lNbElts(List * pl)
{
    return pl->nbe;
}

/* List is not modified (first element is 0th)
 * Access is optimized to be in constant time (only if elements are accessed in
 * growing order).
 * So this function is usable in a loop to get each element of a list.
 *
 * Return: the nth element of list pl.
 *
 * Example: nbe=lNbElts(pl);
 *     for(i=0;i<nbe;i++)
 *        doSomething(lLookNth(pl,i));
 */
void *
lLookNth(List * pl, int n)
{
    int nb;
    ListElt *ple;

    if ((n >= pl->lastN) && (pl->modified == False) && (pl->lastAns != NULL)) {
	nb = pl->lastN;
	ple = pl->lastAns;
    }
    else {
	nb = 0;
	ple = pl->pFirst;
    }
    pl->modified = False;
    while (ple != NULL) {
	if (nb == n) {
	    pl->lastAns = ple;
	    pl->lastN = n;
	    return ple->pe;
	}
	ple = ple->next;
	nb++;
    }
    UIError("lLookNth", "no element of this rank %d", n);
    return NULL;			/* not reached */
}

/* Extract first element form list pl which match with key pk
 */
void
lDelKey(List * pl, void *pk, Bools freeElt)
{
    lDel(pl, lLookKey(pl, pk), freeElt);
}

/* Extract first occurence of element pe form list pl
 */
void
lDel(List * pl, void *pe, Bools freeElt)
{
    ListElt *ple, *plb = NULL;

    pl->modified = True;
    ple = pl->pFirst;
    while (ple != NULL) {
	if (ple->pe == pe) {
	    if (plb == NULL)
		pl->pFirst = ple->next;
	    else
		plb->next = ple->next;
	    if (ple == pl->pLast) {
		if (plb != NULL) {
		    pl->pLast = plb;
		}
		else {
		    pl->pLast = pl->pFirst;
		}
	    }
	    pl->nbe--;
	    lFreeListElt(pl, ple, freeElt);
	    break;
	}
	plb = ple;
	ple = ple->next;
    }
}

/* ODOT...
 */
static ListElt *
lLookListElt(List * pl, void *pe)
{
    ListElt *ple;

    ple = pl->pFirst;
    while (ple != NULL) {
	if (ple->pe == pe) {
	    return ple;
	}
	ple = ple->next;
    }
    return (ListElt *) NULL;
}

/* Return: First ListElt* in which pe match key pk.
 */
static ListElt *
lLookListKey(List * pl, void *pk)
{
    ListElt *ple;

    ple = pl->pFirst;
    while (ple != NULL) {
	if (pl->ordFunc(pl->keyFunc(ple->pe), pk) == 0) {
	    return ple;
	}
	ple = ple->next;
    }
    return (ListElt *) NULL;
}


/* Does pl contain at least one element matching with key pk ?
 */
Bools
lInKey(List * pl, void *pk)
{
    return (lLookListKey(pl, pk)) ? True : False;
}

/* Does pl contain at least one element matching with element pe ?
 */
Bools
lIn(List * pl, void *pe)
{
    return (lLookListKey(pl, pl->keyFunc(pe))) ? True : False;
}

/* Return: First element in pl matching key pk.
 */
void *
lLookKey(List * pl, void *pk)
{
    ListElt *ple;

    return (ple = lLookListKey(pl, pk)) ? ple->pe : NULL;
}

/* Return: First element in pl matching with element pe.
 */
void *
lLook(List * pl, void *pe)
{
    ListElt *ple;

    return (ple = lLookListKey(pl, pl->keyFunc(pe))) ? ple->pe : NULL;
}

/* Cut and return head of list pl.
 */
void *
lGetHead(List * pl)
{
    ListElt *ple;
    void *pe;

    ple = pl->pFirst;
    if (ple) {
	pl->pFirst = ple->next;
	if (ple == pl->pLast)		/* last element of list */
	    pl->pLast = NULL;
	pl->nbe--;
	pl->modified = True;
	pe = ple->pe;
	UFree(ple);
	return pe;
    }
    else				/* empty list */
	return NULL;
}

/* Return: return (but don't remove) head of list pl.
 */
void *
lLookHead(List * pl)
{
    return (pl->pFirst) ? pl->pFirst->pe : NULL;
}

/* Mark this element (pe) as to be sweeped by next call to lSweep(pl).
 */
void
lMarkToSweep(List * pl, void *pe)
{
    ListElt *ple;

    if ((ple = lLookListElt(pl, pe)))
	ple->flag = LIST_FLAG_SWEEP;
}


/* Sweep all elements marked by a lMarkToSweep out of list pl.
 *
 * Return: number of elements left (after sweep).
 */
int
lSweep(List * pl,
       Bools freeElt			/* must the element be freed?
					 * (calling the freeElt function
           given at list creation) */ )
{
    ListElt *ple, *pltmp, *plb = NULL;

    pl->modified = True;
    ple = pl->pFirst;
    while (ple != NULL) {
	if (ple->flag == LIST_FLAG_SWEEP) {
	    if (plb == NULL)
		pl->pFirst = ple->next;
	    else
		plb->next = ple->next;
	    if (ple == pl->pLast) {
		if (plb != NULL) {
		    pl->pLast = plb;
		}
		else {
		    pl->pLast = pl->pFirst;
		}
	    }
	    pl->nbe--;
	    pltmp = ple->next;
	    if (freeElt)
		lFreeListElt(pl, ple, freeElt);
	    ple = pltmp;
	}
	else {
	    plb = ple;
	    ple = ple->next;
	}
    }
    return pl->nbe;
}

/* Reverse list in place.
 */
void
lReverse(List * pl)
{
    ListElt *p, *pp, *pn;

    pl->modified = True;
    pl->pLast = pl->pFirst;
    p = pl->pFirst;
    pp = NULL;
    while (p != NULL) {
	pn = p->next;
	p->next = pp;
	pp = p;
	p = pn;
    }
    pl->pFirst = pp;
}


/* Print list using the print function given at pl creatin.
 *
 * Useful for debuging.
 */
void
lPrint(List * pl)
{
    ListElt *p;

    printf("((%d :%d elts at x%x):\n  ", pl->number, pl->nbe, (unsigned long) pl);
    p = pl->pFirst;
    while (p != NULL) {
	(*(pl->printFunc)) (p->pe);
	p = p->next;
	if (p != NULL)
	    printf("->");
    }
    printf(")\n");
}
