#include <fcntl.h>
#include <signal.h>
#include <sys/wait.h>

#include "util.h"

/* protect mask for created files */
#define MASK 0666


/**************************************************************** Prototypes */

/* Convert a compressed file (failed to use) into a non compressed file
 */
static FILE *
     UCtoNC(char *fileNameC, char *fileNameNC, FILE * f);

/* ODOT...
 */
static FILE *
     UCompressedFCreate(char *fileNameC, char *fileNameNC, int *ppid);

/* ODOT...
 */
static FILE *
     UCompressedFRead(char *fileNameC, char *fileNameNC, int *ppid);

/******************************************************************** Bodies */


/* Convert a compressed file (failed to use) into a non compressed file
 */
static FILE *
UCtoNC(char *fileNameC, char *fileNameNC, FILE * f)
{
    fclose(f);
    if ((f = fopen(fileNameNC, "w")) == NULL) {
	UIWarning("UCompressedFCreate", "cannot open file %s for creation! unable to recover this error!", fileNameNC);
	return (FILE *) NULL;
    }
    return f;
}

/* ODOT...
 */
static FILE *
UCompressedFCreate(char *fileNameC, char *fileNameNC, int *ppid)
{
    int p[2], fdc;
    FILE *fileC;

    if ((fdc = open(fileNameC, O_WRONLY | O_CREAT | O_TRUNC, MASK)) == -1) {
	UIWarning("UCompressedFCreate", "cannot open file %s for creation!", fileNameC);
	return (FILE *) NULL;
    }

    if (!(fileC = fdopen(fdc, "w"))) {
	UIWarning("UCompressedFCreate", "cannot fdopen file %s (%d)!", fileNameC, fdc);
	return (FILE *) NULL;
    }

    if (pipe(p)) {
	UIWarning("UCompressedFCreate", "cannot open pipe for compression!");
	fclose(fileC);
	return (FILE *) NULL;
    }

    switch (*ppid = fork()) {
    case -1:{				/* failed fork() */
	    UIWarning("UCompressedFCreate", "cannot vfork to execute %s, file won't be compressed.", COMPRESSOR);
	    return UCtoNC(fileNameC, fileNameNC, fileC);
	    break;
	}
    case 0:{				/* child process */
	    close(0);
	    dup(p[0]);
	    close(1);
	    dup(fdc);
	    close(p[0]);
	    close(fdc);
	    close(p[1]);
	    if ((execlp(COMPRESSOR, COMPRESSOR_OPT1, (char *) 0)) == -1) {
		UIError("UCompressedFCreate", "cannot find %s in your PATH!", COMPRESSOR);
		exit(1);		/* son kill self */
	    }
	    break;
	}
    default:{				/* main process */
	    if (!(fileC = fdopen(p[1], "w"))) {
		UIWarning("UCompressedFCreate", "cannot fdopen pipe (%d)! file won't be compressed", p[1]);
		kill(*ppid, SIGHUP);	/* kill son */
		return UCtoNC(fileNameC, fileNameNC, fileC);
	    }
	    close(p[0]);
	    close(fdc);
	    return fileC;
	}
    }
    return (FILE *) NULL;
}


/* ODOT...
 */
static FILE *
UCompressedFRead(char *fileNameC, char *fileNameNC, int *ppid)
{
    int p[2], fdc;
    int compressed = True;
    FILE *fileUNC;

    if (((fdc = open(fileNameC, O_RDONLY)) == -1) &&
	((fdc = open(fileNameNC, O_RDONLY), compressed = False) == -1)) {
	UIWarning("UCompressedFRead", "cannot open file %s or %s to read!", fileNameC, fileNameNC);
	return (FILE *) NULL;
    }

    /* non compressed file */
    if (!compressed) {
	if (!(fileUNC = fdopen(fdc, "r"))) {
	    UIWarning("UCompressedFRead", "cannot fdopen %s (%d)!", fileNameC, p[0]);
	    return (FILE *) NULL;
	}
	return fileUNC;
    }

    /* compressed file */
    if (pipe(p)) {
	UIWarning("UCompressedFRead", "cannot open pipe for compression... sorry this won't work! unable to read!");
	return (FILE *) NULL;
    }

    switch (*ppid = fork()) {
    case -1:{				/* failed fork() */
	    UIWarning("UCompressedFRead", "cannot vfork to execute %s... sorry this won't work! unable to read!", UNCOMPRESSOR);
	    return (FILE *) NULL;
	    break;
	}
    case 0:{				/* child process */
	    close(0);
	    dup(fdc);
	    close(1);
	    dup(p[1]);
	    close(p[0]);
	    close(fdc);
	    close(p[1]);
	    if ((execlp(UNCOMPRESSOR, UNCOMPRESSOR_OPT1, UNCOMPRESSOR_OPT2, (char *) 0)) == -1) {
		UIError("UCompressedFRead", "cannot find %s in your PATH... sorry this won't work! unable to read %s!", UNCOMPRESSOR, fileNameC);
		exit(1);
	    }
	    break;
	}
    default:{				/* main process */
	    if (!(fileUNC = fdopen(p[0], "r"))) {
		UIWarning("UCompressedFRead", "cannot fdopen pipe (%d)!", p[0]);
		kill(*ppid, SIGHUP);	/* kill son */
		return (FILE *) NULL;
	    }
	    close(p[1]);
	    close(fdc);
	    return fileUNC;
	}
    }
}

/* Open a file for compressed writting or uncompressed reading.
 *
 * File can be closed with fclose()
 */
FILE *
UCompressedFOpen(char *fileName,
		 char *type /* "w" -> write, "r" -> read */ )
{
    char fileNameC[MAX_FILE_LEN], fileNameNC[MAX_FILE_LEN], suffix[MAX_FILE_LEN];
    FILE *f;
    int pid = 0;

    strcpy(fileNameC, fileName);
    strcpy(fileNameNC, fileName);
    UGetSuffix(fileName, suffix);
    if (!strcmp(suffix, COMPRESSOR_SUFFIX)) {
	UExtractSuffix(fileNameNC, COMPRESSOR_SUFFIX);
    }
    else {
	UAddSuffix(fileNameC, COMPRESSOR_SUFFIX);
    }

    if (!strcmp(type, "w")) {
	f = UCompressedFCreate(fileNameC, fileNameNC, &pid);
	return f;
    }
    else if (!strcmp(type, "r")) {
	f = UCompressedFRead(fileNameC, fileNameNC, &pid);
	return f;
    }
    else
	UIWarning("UCompressedFOpen", "error, can be used only for \"r\" or \"w\" modes! no %s mode suported.", type);
    return (FILE *) NULL;
}
