/* Copyright 1989-93 GROUPE BULL -- See license conditions in file COPYRIGHT */
/***********************\
* 		        *
*  KlO: Number  *
*  BODY		        *
* 		        *
\***********************/

#include "EXTERN.h"
#include "klone.h"
#include "kl_atom.h"
#include "kl_string.h"
#include "kl_list.h"
#include "INTERN.h"
#include "kl_number.h"

KlNumber KlNumberRawMake();
KlO KlRealAdd();

/*
 * Constructor: KlNumberMake
 *   argument 1: The Int (int of size char *) which is to be our number.
 */

KlNumber
KlNumberMake(i)
    Int i;				/* just the Int value of the integer */
{
    KlNumber object;

    if (i <= MAX_BUILT_IN_NUMBER && i >= MIN_BUILT_IN_NUMBER)
	return KlBuiltInNumbers[i - MIN_BUILT_IN_NUMBER];
    object = (KlNumber) KlOMake(KlNumberType);
    object->number = i;
    return object;
}

KlNumber
KlNumberRawMake(i)
    long i;				/* just the LONG value of the integer */
{
    KlNumber object = (KlNumber) KlOMake(KlNumberType);

    object->number = i;
    return object;
}

/*
 * KlNumberPrint:
 * a number prints as a long.
 */

KlO
KlNumberPrint(obj, stream)
    KlNumber obj;
    KlO stream;
{
    KlSPrintf(stream, "%ld", obj->number);
    return (KlO) obj;
}

/*
 * KlNumberFree:
 * The structure is just freed.
 */

KlO
KlNumberFree(obj)
    KlNumber obj;
{
    Free(obj);
    return (KlO) obj;
}

KlO
KlNumberEqual(n1, n2)
    KlNumber n1, n2;
{
    if (KlIsANumber(n2)) {
	if (KlIsAReal(n2)) {
	    if (((KlReal) n2)->real == n1->number)
		return (KlO) n1;
	    else
		return NIL;
	} else {
	    if (n2->number == n1->number)
		return (KlO) n1;
	    else
		return NIL;
	}
    } else {
	return NIL;
    }
}

KlO
KlNumberAdd(argc, argv)
    int argc;
    KlNumber argv[];

{
    int i, result = argv[0]->number;

    for (i = 1; i < argc; i++) {
	KlMustBeNumber(argv[i], i);
	if (KlIsAReal(argv[i])) {
	    KlReal r = (KlReal) KlRealAdd(argc - i, argv + i);

	    r->real += result;
	    KlRealFix(r);
	    return (KlO) r;
	} else {
	    result += argv[i]->number;
	}
    }
    return (KlO) KlNumberMake(result);
}

/* KlNumberCoerce
 * string ==> does a scanf (atoi)
 */

/*ARGSUSED*/
KlO
KlNumberCoerce(totype, obj)
    KlType totype;
    KlString obj;
{
    if (KlIsAString(obj)) {
	long n = 0;
	if ((obj->string[0] == '0'
	     && (obj->string[1] == 'x' || obj->string[1] == 'X')
	     && sscanf(obj->string + 2, "%lx", &n))
	     ||
	     sscanf(obj->string, "%ld", &n)) {
	    return (KlO) KlNumberMake(n); /* Ok */
	} else {
	    return 0;
	}
    } else if (KlIsAReal(obj)) {
	return (KlO) KlNumberMake(((KlNumber) obj)->number);
    } else if (KlIsANumber(obj))
	return (KlO) obj;
    return 0;
}

KlO
KlNumberHash(obj)
    KlNumber obj;
{
    return (KlO) obj->number;
}


/* (N obj) ==> (getn obj N)
 * (N obj val) ==> (put obj N val)
 */

KlO
KlNumberExecuteOrApply(num, list, eval)
    KlNumber num;
    KlList list;
    int eval;				/* eval args? */
{
    KlO obj = list->list[1];

    if (list->size == 2) {		/* get */
	if (eval)
	    obj = KlSend_eval(obj);
	return KlSend_nth(obj, num->number, 0);
    } else if (list->size == 3) {	/* put */
	KlO val = list->list[2];
	if (eval) {
	    obj = KlSend_eval(obj);
	    val = KlSend_eval(val);
	}
        return KlSend_nth(obj, num->number, val);
    } else {
	return CFAPPLY((KlSelectorUndefmethod(eval ? KlSelExecute : KlSelApply)), 
	    (num, list));
    }
}

KlO
KlNumberExecute(num, list)
    KlNumber num;
    KlList list;
{
    return KlNumberExecuteOrApply(num, list, 1);
}

KlO
KlNumberApply(num, list)
    KlNumber num;
    KlList list;
{
    return KlNumberExecuteOrApply(num, list, 0);
}


int
KlNumberCompare(o1, o2)
    KlNumber o1, o2;
{
    if (KlIsANumber(o2)) {
	if (KlIsAReal(o2)) {
	    return KlRealCompare(o1, o2);
	} else {
	    return o1->number - o2->number;
	}
    } else {
	return (int) KlBadArgument(o1, 0, KlTypeCName(KlNumberType));
    }
}

/* needed from parser */
KlO
KlUnsignedMakeFromString(s)
    char *s;
{
    Int n;
    sscanf(s+2, "%lu", &n);
    return (KlO) KlNumberMake(n);
}

/*****************************************************************************\
* 				 Real numbers                                 *
\*****************************************************************************/

KlReal
KlRealMake(x)
    double x;
{
    KlReal obj = (KlReal) KlOMake(KlRealType);

    obj->real = x;
    KlRealFix(obj);
    return obj;
}

/* print of a real
 * in natural format, but always with a dot if we print readably
 */

#define KlMAXREALPRECISION 20
#define KlMAXREALPRECISIONFORMAT "%.20g"

KlO
KlRealPrint(obj, stream)
    KlReal obj;
    KlO stream;
{
    char s[KlMAXREALPRECISION+12];

    if (KlPrintReadably || KlRealPrecision >= KlMAXREALPRECISION) {
	sprintf(s, KlMAXREALPRECISIONFORMAT, obj->real); 
    } else if (KlRealPrecision < 0) {
	sprintf(s, "%lg", obj->real);
    } else {
	char format[10];
	sprintf(format, "%%.%dlg", KlRealPrecision);
	sprintf(s, format, obj->real);
    }
    if (!strchr(s, '.') && !strchr(s, 'e')) { /* prints as an integer */
	strcat(s, ".0");		/* adds a dot to show it is a real */
    }
    KlSPuts(s, stream);
    return (KlO) obj;
}

KlO
KlRealEqual(n1, n2)
    KlReal n1, n2;
{
    if (KlIsAReal(n2)) {
	if (n2->real == n1->real)
	    return (KlO) n1;
	else
	    return NIL;
    } else if (KlIsANumber(n2)) {
	if (((KlNumber) n2)->number == n1->real)
	    return (KlO) n1;
	else
	    return NIL;
    } else
	return NIL;
}


KlO
KlRealAdd(argc, argv)
    int argc;
    KlReal argv[];

{
    int i;
    double result = argv[0]->real;

    for (i = 1; i < argc; i++) {
	KlMustBeNumber(argv[i], i);
	result += KlNumberRealValue(argv[i]);
    }
    return (KlO) KlRealMake(result);
}

/* KlRealCoerce
 * string ==> does a scanf (atoi)
 */

/*ARGSUSED*/
KlO
KlRealCoerce(totype, obj)
    KlType totype;
    KlString obj;
{
    if (KlIsAString(obj)) {
	double x;
	long n;
	if (obj->string[0] == '0'
	    && (obj->string[1] == 'x' || obj->string[1] == 'X')
	    && sscanf(obj->string + 2, "%lx", &n)) {
	    return (KlO) KlRealMake((double) n);
	} else if (sscanf(obj->string, "%lf", &x)) {
	    return (KlO) KlRealMake(x); /* Ok */
	} else {
	    return 0;
	}
    } else if (KlIsAReal(obj)) {
	return (KlO) obj;
    } else if (KlIsANumber(obj)) {
	return (KlO) KlRealMake((double) (((KlNumber) obj)->number));
    }
    return 0;
}

KlO
KlNumbersCoerce(totype, obj)
    KlType totype;
    KlString obj;
{
    if (KlIsAString(obj)) {
	if (strchr(obj->string, '.')
	    || strchr(obj->string, 'e')) { /* prints as an integer */
	    return KlRealCoerce(totype, obj);
	}
    }
    /* else just like an int */
    return KlNumberCoerce(totype, obj);
}


KlO
KlRealHash(obj)
    KlReal obj;
{
    unsigned int result = obj->real + obj->real * 1000000;
    return (KlO) result;
}

int
KlRealCompare(o1, o2)
    KlReal o1, o2;
{
    if (KlIsANumber(o2)) {
	double r1 = KlNumberRealValue(o1);
	double r2 = KlNumberRealValue(o2);

	if (r1 > r2)
	    return 1;
	else if (r1 < r2)
	    return -1;
	else
	    return 0;
    } else {
	return (int) KlBadArgument(o1, 0, KlTypeCName(KlNumbersType));
    }
}

/* elevation to integral powers: **
 */

KlO
KlPower(x, n)
    KlReal x;
    KlNumber n;
{
    int elevation;

    KlMustBeInteger(n, 1);
    if (n->number >= 0)
	elevation = n->number;
    else
	elevation = - n->number;
    if (KlIsAReal(x)) {
	double val = 1.0;
	while (elevation--)
	    val *= x->real;
	if (n->number < 0)
	    val = 1 / val;
	return (KlO) KlRealMake(val);
    } else if (KlIsAnInteger(x)) {
	int val = 1;
	while (elevation--)
	    val *= x->number;
	if (n->number < 0) {
	    double result = 1 / ((double) val);
	    return (KlO) KlRealMake(result);
	} else {
	    return (KlO) KlNumberMake(val);
	}	
    }
    KlMustBeNumber(x, 0);
    /* NOTREACHED */
    return 0;
}

/* (incf var value)
 * increments variable by value
 */

KlO
KlIncf(argc, argv)
    int argc;
    KlO *argv;
{
    KlO values[2];
    KlO newval;

    if (argc == 0 || argc > 2)
	return KlBadNumberOfArguments((char *) argc);
    values[0] = KlSend_eval(argv[0]);
    if (argc == 2)
	values[1] = KlSend_eval(argv[1]);
    else
	values[1] = (KlO) KlBuiltInNumbers[1 - MIN_BUILT_IN_NUMBER];
    newval = KlSendNary(KlSelAdd, 2, values);
    return KlSend_setq(argv[0], newval);
}


/*****************************************************************************\
* 				  TYPE INIT                                   *
\*****************************************************************************/

KlNumberInit()
{
    KlDeclareType(&KlMagnitudeType, "Magnitude", 0);
    KlDeclareType(&KlNumbersType, "Number", 0);
    KlTypeFatherSet(KlNumbersType, KlMagnitudeType);
    KlDeclareIsTrait(KlNumbersType, KlTrait_number);

    KlDeclareSubType(&KlNumberType, "Int", KlNumbersType,
		     sizeof(struct _KlNumber));
    KlA_Int = KlTypeName(KlNumberType);

    KlDeclareMethod1(KlNumberType, KlSelPrint, KlNumberPrint);
    KlDeclareMethod1(KlNumberType, KlSelEqual, KlNumberEqual);
    KlDeclareMethod1(KlNumberType, KlSelAdd, KlNumberAdd);
    KlDeclareMethod1(KlNumberType, KlSelHash, KlNumberHash);
    KlDeclareMethod1(KlNumberType, KlSelCompare, (KlMethod) KlNumberCompare);
    KlDeclareMethod1(KlNumberType, KlSelExecute, KlNumberExecute);
    KlDeclareMethod1(KlNumberType, KlSelApply, KlNumberApply);
    KlDeclareSubType(&KlRealType, "Real", KlNumbersType,
		     sizeof(struct _KlReal));

    KlDeclareMethod1(KlRealType, KlSelPrint, KlRealPrint);
    KlDeclareMethod1(KlRealType, KlSelEqual, KlRealEqual);
    KlDeclareMethod1(KlRealType, KlSelAdd, KlRealAdd);
    KlDeclareMethod1(KlRealType, KlSelHash, KlRealHash);
    KlDeclareMethod1(KlRealType, KlSelCompare, (KlMethod) KlRealCompare);

    KlDeclareSubr(KlPower, "**", 2);
    KlDeclareFSubr(KlIncf, "incf", NARY);


    {					
	/* pre-calculate "low" numbers (most used ones) */
	long i;
	KlNumber *p = KlBuiltInNumbers;

	for (i = MIN_BUILT_IN_NUMBER; i <= MAX_BUILT_IN_NUMBER; i++, p++)
	    KlIncRef((*p) = KlNumberRawMake(i));
    }
}
