#ifndef EMTREE_H
#define EMTREE_H

/* a special node which is always present */
#define EmGateNode ((EmNode*) NULL)


typedef enum EmNodeType {
    NTBad = 0, NTPlus, NTMult, NTDiv, NTExp, NTSub,
    NTUMinus, NTEq,
    NTRoot, NTPow, NTIntegral,
    NTFunc,
    NTColumn, NTVector, NTMatrix,
    NTString, NTNumber, NTLetter, NTDigit, NTSymbol,
    NTParenth, NTTemplate, NTContainer
}          EmNodeType;

/* struct of a Formula Tree Node */
typedef struct EmNode {
    struct EmNode *pf;			/* father */
    int level;				/* level in tree (root = level 0) */
    union {
	struct EmNode **aps;		/* array of sons */
	KlO value;			/* value for leaf */
    }     d;
    EmNodeType type;			/* type of node */
    Byte nbsn;				/* NumBer of Son Nodes in aps 0 mean
					 * that it's a leaf so d.value must
					 * be used instead of d.aps */
}      EmNode;



/**************************************************************** Prototypes */


/* Wrapper: give a node and return a human readable type
 */
KlO EmNodeTypeToStringKl ( KlO idn );

/* Convert a node type to its human readable form :)
 */
char * EmNodeTypeToString ( EmNodeType type );

/* Wrapper: give a char and return its readable type
 */
KlO EmKeyToReadableNodeTypeKl ( KlO key );

/* Wrapper: give a char and return its type
 */
KlO EmKeyToNodeTypeKl ( KlO key );

/* Print a node
 */
void EmPrintEmNode ( EmNode * pn );

/* Print a node tree
 */
void EmPrintTree ( EmNode * pn );

/* Wrapper:
 */
KlO EmPrintTreeKl ( KlO id );

/* Type of node the created node. */
EmNode * EmCreateNode ( EmNodeType type );

/* Wrapper:
 */
KlO EmCreateNodeKl ( KlO type );

/* Delete a tree (pn)
 *
 * Delete the node pn and all its descendants.
 * This node is discarded from its father array of sons (which is not compacted).
 * Warning: The attributes pined to this tree must be explicitly freed.
 */
void EmDelTree ( EmNode * pn );

/* Wrapper:
 */
KlO EmDelTreeKl ( KlO id );

/* The node to delete */
void EmDelNode ( EmNode * pn );

/* Wrapper:
 */
KlO EmDelNodeKl ( KlO id );

/* is added as last son of pn */
int EmAddSonNode ( EmNode * pn, EmNode * ps );

/* Wrapper:
 */
KlO EmAddSonNodeKl ( KlO idn, KlO ids );

/* Add the Nth (num) Son (ps) to a Node (pn).
 *
 * Current sons are right shifted if needed
 */
void EmAddSonNodeNth ( EmNode * pn, EmNode * ps, int num );

/* Wrapper:
 */
KlO EmAddSonNodeNthKl ( KlO idn, KlO ids, KlO num );

/* Replace the Nth (num) Son (ps) to a Node (pn)
 *
 * Current son number (num) and all is descendants are freed with EmDelTree()
 */
void EmReplaceSonNodeNth ( EmNode * pn, EmNode * ps, int num );

/* Wrapper:
 */
KlO EmReplaceSonNodeNthKl ( KlO idn, KlO ids, KlO num );

/* Add an array (aps) of (num) sons to a node (pn)
 *
 * Warning: Array aps must be freed by caller
 *
 * Return: number of sons
 */
int EmAddSonNodes ( EmNode * pn, EmNode ** aps, int num );

/* Wrapper:
 */
KlO EmAddSonNodesKl ( KlO idn, KlO aps, KlO num );

/* Remove a son (ps) from node (pn)
 *
 * Nothing is freed (tree (ps) may be linked to another father).
 * However ps may be freed explicitly (eg: with EmDelNode or EmDelTree)
 */
void EmRemoveSonNode ( EmNode * pn, EmNode * ps );

/* Wrapper:
 */
KlO EmRemoveSonNodeKl ( KlO idn, KlO ids );

/* Remove the Nth (num) son (ps) from node (pn).
 */
void EmRemoveSonNodeNth ( EmNode * pn, int num );

/* Wrapper:
 */
KlO EmRemoveSonNodeNthKl ( KlO idn, KlO num );

/* Given a node compute its rank as a son
 * return -1 if the son is not found as a child of this father
 */
int EmFindRankOfSon ( EmNode * ps );

/* Wrapper:
 */
KlO EmFindRankOfSonKl ( KlO ids );

/* Insert a node (pi) between a father (pf) and its son (ps)
 * thus adding 1 to the tree level of the son
 */
EmNode * EmInsertNode ( EmNode * pf, EmNode * ps, EmNode * pi );

/* Wrapper:
 */
KlO EmInsertNodeKl ( KlO idf, KlO ids, KlO idi );

/* Insert a node (pi) between a father (pf) and its son (ps)
 * thus adding 1 to the tree level of the son.
 *
 * Warning: Not tested.
 */
EmNode * EmInsertNodeNth ( EmNode * pf, EmNode * ps, EmNode * pi, int num );

/* Wrapper:
 */
KlO EmInsertNodeNthKl ( KlO idf, KlO ids, KlO idi, KlO num );

/* Change the level of all nodes in a tree (pn).
 *
 * NewLevel becomes the root level and all subtree nodes are updated acordingly
 */
void EmChangeTreeLevel ( EmNode * pn, int newLevel );

/* Wrapper:
 */
KlO EmChangeTreeLevelKl ( KlO idn, KlO newLevel );

/* Compact the sons of a node
 *
 * Remove empty sons from array of sons
 *
 * Return: the number of non empty sons
 */
int EmCompactSonNodes ( EmNode * pn );

/* Wrapper:
 */
KlO EmCompactSonNodesKl ( KlO idn );

/* The node */
EmNodeType EmTypeOfNode ( EmNode * pn );

/* Wrapper:
 */
KlO EmTypeOfNodeKl ( KlO idn );

/* The new type */
EmNode * EmSetNodeType ( EmNode * pn, EmNodeType type );

/* Wrapper:
 */
KlO EmSetNodeTypeKl ( KlO idn, KlO type );

/* The node */
EmNode * EmFatherNode ( EmNode * pn );

/* Wrapper:
 */
KlO EmFatherNodeKl ( KlO idn );

/* The node */
KlO EmGetNodeValue ( EmNode * pn );

/* Wrapper:
 */
KlO EmGetNodeValueKl ( KlO idn );

/* The value to store */
KlO EmSetNodeValue ( EmNode * pn, KlO val );

/* Wrapper:
 */
KlO EmSetNodeValueKl ( KlO idn, KlO val );

/* The node */
int EmNbSonNodes ( EmNode * pn );

/* Wrapper:
 */
KlO EmNbSonNodesKl ( KlO idn );

/* Rank of node to return */
EmNode * EmNthSonNode ( EmNode * pn, int n );

/* Wrapper:
 */
KlO EmNthSonNodeKl ( KlO idn, KlO num );

/* Initialize all tree manipulating functions for Klone
 */
void EmTreeInit ( void );

/************************************************************ End Prototypes */


#endif					/* EMTREE_H */
